/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

//  test07.cpp -- cluster tests

// In this example
// mod  : t_mod = t_in/out * rate
//
// mod_a: 1.0 =  1.0 * 10
// mod_b: 1.0 =  1.0 * 5

// results in cluster execution schedule { A B B }

#include <systemc-ams>
#include "test_utilities.h"

//#define TDF_DEBUG

SCA_TDF_MODULE( mod_a )
{
  sca_tdf::sca_out<double> out;

  void processing()
  {
    std::cout << get_time() << ": A" << std::endl;

    for (int i = 0; i < rate;  i++)
    {
      out.write(val, i);
#ifdef TDF_DEBUG
      std::cout << out.get_time(i) << ": A.out(" << i << ") = " << val << std::endl;
#endif
      val++;
    }
  }

  void set_attributes()
  {
    out.set_timestep(1.0, sc_core::SC_MS);
    out.set_rate(rate);
  }

  SCA_CTOR( mod_a ) : out("out"), val(5.5), rate(10)
  {}

 private:
  int rate;
  double val;
};

SCA_TDF_MODULE( mod_b )
{
  sca_tdf::sca_in<double> in;
  sca_tdf::sca_de::sca_in<double> in_de;

  void processing()
  {
    std::cout << get_time() << ": B" << std::endl;

    double val_de = in_de.read();
#ifdef TDF_DEBUG
    std::cout << in_de.get_time() << ": B.in_de = " << val_de << std::endl;
#endif

    for (int i = 0; i < rate;  i++)
    {
      double val = in.read(i);
#ifdef TDF_DEBUG
    std::cout << in.get_time(i) << ": B.in(" << i << ") = " << val << std::endl;
#endif
    }
  }

  void set_attributes()
  {
    in.set_timestep(1.0, sc_core::SC_MS);
    in.set_rate(rate);
  }

  SCA_CTOR( mod_b ) : in("in"), in_de("in_de"), rate(5)
  {}

 private:
  int rate;
};

SC_MODULE( mod_d )
{
  sc_core::sc_out<double> out;

  double val0, val1, val2;

  void gen_val()
  {
    while (true)
    {
      out.write(val0);
#ifdef TDF_DEBUG
      std::cout << sc_core::sc_time_stamp() << ": D.out = " << val0 << std::endl;
#endif
      wait(sc_core::sc_time(0.9, sc_core::SC_MS));
      out.write(val1);
#ifdef TDF_DEBUG
      std::cout << sc_core::sc_time_stamp() << ": D.out = " << val1 << std::endl;
#endif
      wait(sc_core::sc_time(1.0, sc_core::SC_MS));
      out.write(val2);
#ifdef TDF_DEBUG
      std::cout << sc_core::sc_time_stamp() << ": D.out = " << val2 << std::endl;
#endif
      wait(sc_core::sc_time(1.0, sc_core::SC_MS));
    }
  }

  SC_CTOR( mod_d ) : out("out"), val0(10.0), val1(20.0), val2(30.0)
  {
    SC_THREAD( gen_val );
  }
};

int sc_main( int, char*[] )
{
  TEST_LABEL_START;

  sca_tdf::sca_signal<double> sig1;
  sc_core::sc_signal<double> sig2;

  mod_a a("a");
   a.out(sig1);

  mod_d d("d");
   d.out(sig2);

  mod_b b("b");
   b.in(sig1);
   b.in_de(sig2);

  sca_util::sca_trace_file* atf = sca_util::sca_create_tabular_trace_file("test07");

  sca_util::sca_trace(atf, sig1, "sig1");
  sca_util::sca_trace(atf, sig2, "sig2");

  sc_core::sc_start(15.01, sc_core::SC_MS);

  sca_util::sca_close_tabular_trace_file(atf);

  test_util::check_results("test07", 2);

  TEST_LABEL_END;

  return 0;
}
