/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/
// test05.cpp  - lsf regression test
//  test05 with one tdf_demux et al.

#include <systemc-ams>
#include "test_utilities.h"
#include <string>
#include <fstream>
#include <sstream>

SC_MODULE(test05)
{
  // port declaration
  sca_lsf::sca_out outp1;
  sca_lsf::sca_out outp2;

  // node declaration
  sca_lsf::sca_signal n5;
  sca_lsf::sca_signal n2;
  sca_tdf::sca_signal<bool> n1;

  // component declaration
  sca_lsf::sca_source *sca_source1;
  sca_lsf::sca_dot *dot1;
  sca_lsf::sca_tdf_demux *tdf_demux1;

  struct params // parameter
  {
    params() // default for parameter
    {
    }
  };

  test05( sc_core::sc_module_name nm, params pa = params() )
  : outp1("outp1"), outp2("outp2"), n5("n5"), n2("n2"), n1("n1"), p(pa)
  {
    architecture(); // generate netlist
  }

  ~test05();

private:
  params p;
  void architecture(); // method for netlist
};

// netlist implementation
void test05::architecture()
{
  sca_source1 = new sca_lsf::sca_source("sca_source1");
  sca_source1 -> init_value = 1.0;
  sca_source1 -> offset = 0.0;
  sca_source1 -> amplitude = 1.0e-2;
  sca_source1 -> frequency = 1.0e-3;
  sca_source1 -> phase = 0.0;
  sca_source1 -> delay = sca_core::sca_time(0.0, sc_core::SC_US);
  sca_source1 -> ac_amplitude = 0.0;
  sca_source1 -> ac_phase = 0.0;
  sca_source1 -> ac_noise_amplitude = 0.0;
  sca_source1 -> y(n5);

  dot1 = new sca_lsf::sca_dot("dot1");
  dot1 -> k = 1.0e-1;
  dot1 -> y(n2);
  dot1 -> x(n5);

  tdf_demux1 = new sca_lsf::sca_tdf_demux("tdf_demux1");
  tdf_demux1 -> y2(outp2);
  tdf_demux1 -> y1(outp1);
  tdf_demux1 -> x(n2);
  tdf_demux1 -> ctrl(n1);
}

test05::~test05()
{
  delete sca_source1, dot1, tdf_demux1;
}

void check_td_results(std::string fname)
{
  bool test_passed_with_success = true;
  std::ifstream tdf(fname.c_str());
  std::string line;
  long test_cnt = 0;

  getline(tdf, line);
  std::cout << "1..7" << std::endl;
  while (getline(tdf, line))
  {
    std::istringstream str(line);

    double time;
    str >> time;

    double outp1, outp2;
    str >> outp1 >> outp2;

    if (time == 0.0)
    {
      if ( !(test_util::check_value(outp1, 0.0) && test_util::check_value(outp2, 0.0)))
      {
        std::cerr << "Error TD-check failed for output1 and output2 at: " << time << std::endl;
        std::cout << "not ";
        test_passed_with_success = false;
      }
      std::cout << "ok " << ++test_cnt << std::endl;
    }

    if ((time > 0.0) && (time < 0.04))
    {
      if ( !(test_util::check_value(outp1, 6.28318530305e-06, 1e-5) && test_util::check_value(outp2, 0.0)))
      {
        std::cerr.precision(15);
        std::cerr << "Error TD-check failed for output1 and output2 at: " << time
            << " outp1 " << outp1 << " outp2: " << outp2  << std::endl;
        std::cout << "not ";
        test_passed_with_success = false;
      }
      std::cout << "ok " << ++test_cnt << std::endl;
    }

    if (time > 0.03)
    {
      if ( !(test_util::check_value(outp1, 0.0) && test_util::check_value(outp2, 6.28318515422e-06, 1e-5)))
      {
        std::cerr << "Error TD-check failed for output1 and output2 at: " << time << std::endl;
        std::cout << "not ";
        test_passed_with_success = false;
      }
      std::cout << "ok " << ++test_cnt << std::endl;
    }
  };

  if (test_passed_with_success) std::cout << "results ok." << std::endl;
}

int sc_main(int argc, char* argv[])
{
  TEST_LABEL_START;

  // define non-conservative signals
  sca_lsf::sca_signal output1;
  sca_lsf::sca_signal output2;

  test05* dut = new test05("dut");
  dut->outp1(output1);
  dut->outp2(output2);

  test_util::pulse_src* src = new test_util::pulse_src("source");
  src -> outp(dut->n1);

  sca_util::sca_trace_file* tf = sca_util::sca_create_tabular_trace_file("output4.dat");
  sca_util::sca_trace(tf, dut->outp1, "outp1");
  sca_util::sca_trace(tf, dut->outp2, "outp2");

  sc_core::sc_start(80.0, sc_core::SC_MS);

  sca_util::sca_close_tabular_trace_file(tf);

  check_td_results("output4.dat");

  TEST_LABEL_END;

  delete dut, src;
  
  return 0;
}
