/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

//  band_pass.cpp  - regression test for different kinds of equation description

#include <systemc-ams>
#include "test_utilities.h"
#include <string>

// lowpass implementation as non-conservative behavioral model with sca_ltf_nd

SCA_TDF_MODULE(band_nd)
{
  // ports
  sca_tdf::sca_in<double> inp;
  sca_tdf::sca_out<double> outp;

  struct params // parameter
  {
    double fc;

    params() // default for parameter
    {
      fc = 1.0e3;
    }
  };

  void ac_processing(); // ac-implementation
  void initialize();    // initialization
  void processing();    // time domain processing method

  params p;

  // constructor
  band_nd( sc_core::sc_module_name nm, params pa = params() )
  : inp("inp"), outp("outp"), ltf1("ltf1"), p(pa)
  { }

  // definition of local variables
private:
  sca_tdf::sca_ltf_nd ltf1;
  sca_util::sca_vector<double> A, B, S;
};

// implementation of methods

void band_nd::initialize()
{
  double rc = 1.0 / (2.0 * M_PI * p.fc);

  B(0) = 0.0;
  B(1) = rc;

  A(0) = 1.0;
  A(1) = 3.0 * rc;
  A(2) = rc * rc;
}

// frequency domain implementation
void band_nd::ac_processing()
{
  sca_ac_analysis::sca_ac(outp) = sca_ac_analysis::sca_ac_ltf_nd(B, A, sca_ac_analysis::sca_ac(inp));
}

// time domain implementation
void band_nd::processing()
{
  outp = ltf1(B, A, S, inp);
}

// lowpass implementation as non-conservative behavioral model with sca_ltf_zp

SCA_TDF_MODULE(band_zp)
{
  // ports
  sca_tdf::sca_in<double> inp;
  sca_tdf::sca_out<double> outp;

  struct params // parameters
  {
    double fc;
    double K;

    params() // default for parameters
    {
      fc = 1.0e3;
      K = 1.0;
    }
  };

  void ac_processing(); // ac-implementation
  void initialize();    // initialization
  void processing();    // time domain processing method

  // constructor
  band_zp( sc_core::sc_module_name nm, params pa = params() )
  : inp("inp"), outp("outp"), ltf1("ltf1"), p(pa)
  {}

  // definition of local variables
private:
  sca_tdf::sca_ltf_zp ltf1;
  sca_util::sca_vector<sca_util::sca_complex> Z, P;
  sca_util::sca_vector<double> S;
  params p;
};

// implementation of methods

// vector initialization
void band_zp::initialize()
{
  // H(s) = K (s-Z(0)) / (s-P(0))

  double rc = 1.0 / (2.0 * M_PI * p.fc);
  double a = 3.0 / 2.0 + sqrt(1.5 * 1.5 - 1.0);

  Z(0) =  0.0;
  P(0) = -a / rc;
  P(1) = -(3.0 - a) / rc;

  p.K  =  1.0 / rc;
}

// frequency domain
void band_zp::ac_processing()
{
  sca_ac_analysis::sca_ac(outp) = sca_ac_analysis::sca_ac_ltf_zp(Z, P, sca_ac_analysis::sca_ac(inp), p.K);
}

// time domain
void band_zp::processing()
{
  outp = ltf1(Z, P, S, inp, p.K);
}

// lowpass implementation as hierarchical module with a non-conservative inport
// and a conservative (electrical) outport and implemented as conservative RC-network

SC_MODULE(band_rc)
{
  // port and terminal declaration
  sca_tdf::sca_in<double> inp;
  sca_eln::sca_terminal outp;

  // component declaration
  sca_eln::sca_tdf_vsource* vin;
  sca_eln::sca_r* r1;
  sca_eln::sca_c* c1;
  sca_eln::sca_r* r2;
  sca_eln::sca_c* c2;

  //node declaration
  sca_eln::sca_node n_in, n_out_hp;
  sca_eln::sca_node_ref gnd; // reference node

  struct params // parameter
  {
    double r_val;
    double c_val;

    params() // default for parameter
    {
      double fc = 1.0e3; //default for 1kHz cut off freq.
      r_val = 1.0e3;
      c_val = 1.0/(2.0 * M_PI * fc * r_val);
    }
  };

  band_rc( sc_core::sc_module_name nm, params pa = params() )
  : inp("inp"), outp("outp"), n_in("n_in"), n_out_hp("n_out_hp"),
    gnd("gnd"), p(pa)
  {
    architecture(); //generate netlist
  }

  ~band_rc();

private:
  params p;
  void architecture(); //method for netlist
};

//netlist implementation
void band_rc::architecture()
{
  vin = new sca_eln::sca_tdf_vsource("vin");
  vin->inp(inp);
  vin->p(n_in);
  vin->n(gnd);

  r1 = new sca_eln::sca_r("r1");
  r1->value = p.r_val;           // the value can be assigned via the member...
  r1->n(n_out_hp);
  r1->p(gnd);

  c1 = new sca_eln::sca_c("c1", p.c_val); // or via the constructor
  c1->p(n_in);
  c1->n(n_out_hp);

  r2 = new sca_eln::sca_r("r2");
  r2->value = p.r_val;           // the value can be assigned via the member...
  r2->n(n_out_hp);
  r2->p(outp);

  c2 = new sca_eln::sca_c("c2", p.c_val); // or via the constructor
  c2->p(outp);
  c2->n(gnd);
}

band_rc::~band_rc()
{
  delete vin, r1, c1, r2, c2;
}

// lowpass implementation as hierarchical module with a LSF

SC_MODULE(band_lsf_nd)
{
  // port declaration
  sca_tdf::sca_in<double> inp;
  sca_lsf::sca_out outp;

  // component declaration
  sca_lsf::sca_tdf_source* tdf2lsf;
  sca_lsf::sca_ltf_nd* ltf_nd;

  // node declaration
  sca_lsf::sca_signal s1;

  struct params // parameters
  {
    double fc;

    params() // default for parameters
    {
      fc = 1.0e3; // default for 1kHz cut off freq.
    }
  };

  band_lsf_nd( sc_core::sc_module_name nm, params pa = params() )
  : inp("inp"), outp("outp"), s1("s1"), p(pa)
  {
    architecture(); // generate netlist
  }

  ~band_lsf_nd();

private:
  params p;
  void architecture(); // method for netlist
};

// netlist implementation
void band_lsf_nd::architecture()
{
  tdf2lsf = new sca_lsf::sca_tdf_source("tdf2lsf");
  tdf2lsf->inp(inp);
  tdf2lsf->y(s1);

  sca_util::sca_vector<double> A, B;
  double rc = 1.0/(2.0 * M_PI * p.fc);

  B(0) = 0.0;
  B(1) = rc;

  A(0) = 1.0;
  A(1) = 3.0 * rc;
  A(2) = rc * rc;

  ltf_nd = new sca_lsf::sca_ltf_nd("ltf_nd", B, A);
  ltf_nd->x(s1);
  ltf_nd->y(outp);
}

band_lsf_nd::~band_lsf_nd()
{
  delete tdf2lsf, ltf_nd;
}

// lowpass implementation as hierarchical module with a LSF wit zero-pole

SC_MODULE(band_lsf_zp)
{
  // port declaration
  sca_tdf::sca_in<double> inp;
  sca_lsf::sca_out outp;

  // component declaration
  sca_lsf::sca_tdf_source* tdf2lsf;
  sca_lsf::sca_ltf_zp* ltf_zp;

  // node declaration
  sca_lsf::sca_signal s1;

  struct params // parameters
  {
    double fc;

    params() // default for parameters
    {
      fc = 1.0e3; // default for 1kHz cut off freq.
    }
  };

  band_lsf_zp( sc_core::sc_module_name nm, params pa = params() )
  : inp("inp"), outp("outp"), s1("s1"), p(pa)
  {
    architecture(); // generate netlist
  }

  ~band_lsf_zp();

private:
  params p;
  void architecture(); // method for netlist
};

// netlist implementation
void band_lsf_zp::architecture()
{
  tdf2lsf = new sca_lsf::sca_tdf_source("tdf2lsf");
  tdf2lsf->inp(inp);
  tdf2lsf->y(s1);

  sca_util::sca_vector<sca_util::sca_complex> Z, P;
  double K;

  // H(s) = K (s-Z(0)) / (s-P(0))
  double rc = 1.0 / (2.0 * M_PI * p.fc);
  double a = 3.0 / 2.0 + sqrt(1.5 * 1.5 - 1.0);

  Z(0) =  0.0;
  P(0) = -a/rc;
  P(1) = -(3.0-a)/rc;
  K    =  1.0/rc;

  ltf_zp = new sca_lsf::sca_ltf_zp("ltf_zp", Z, P, K);
  ltf_zp->x(s1);
  ltf_zp->y(outp);
}

band_lsf_zp::~band_lsf_zp()
{
  delete tdf2lsf, ltf_zp;
}

// non-conservative sine wave source

SCA_TDF_MODULE(sin_src)
{
  sca_tdf::sca_out<double> outp;

  struct params
  {
    double freq;
    double ampl;
    sc_core::sc_time sample_time;

    params()
    {
      ampl = 1.0;
      freq = 1e3;
      sample_time = sc_core::sc_time(10.0, sc_core::SC_US);
    }
  };

  void set_attributes()
  {
    outp.set_timestep(p.sample_time);
  }

  void ac_processing();
  void processing();

  sin_src( sc_core::sc_module_name nm, params pa = params() )
  : outp("outp"), p(pa)
  {}

private:
  params p;
};

// frequency domain implementation
void sin_src::ac_processing()
{
  // we use for ac-domain for all frequencies the same amplitude and a phase=0
  sca_ac_analysis::sca_ac(outp) = p.ampl;
}

// time domain implementation
void sin_src::processing()
{
  outp = p.ampl * std::sin(2.0 * M_PI * p.freq * sc_core::sc_time_stamp().to_seconds());
}

// test bench implementation

void check_td_results(std::string fname);
void check_ac_results(std::string fname);

int sc_main(int argn, char* argc[])
{
  TEST_LABEL_START;

  // define non-conservative signals
  sca_tdf::sca_signal<double> s_in, s_out_nd, s_out_zp;

  // define electrical node
  sca_eln::sca_node n_out_rc;

  // lsf signal
  sca_lsf::sca_signal s_lsf_nd_out, s_lsf_zp_out;

  // instantiate and connect components

  sin_src* i_src = new sin_src("i_src");
  i_src->outp(s_in);

  ////////////////////////////////////////////////

  band_nd* i_bp_nd = new band_nd("i_bp_nd");
  i_bp_nd->inp(s_in);
  i_bp_nd->outp(s_out_nd);

  ////////////////////////////////////////////////

  band_zp* i_bp_zp = new band_zp("i_bp_zp");
  i_bp_zp->inp(s_in);
  i_bp_zp->outp(s_out_zp);

  ////////////////////////////////////////////////

  band_rc* i_bp_rc = new band_rc("i_bp_rc");
  i_bp_rc->inp(s_in);
  i_bp_rc->outp(n_out_rc);

  ////////////////////////////////////////////////

  band_lsf_nd* i_bp_lsf_nd = new band_lsf_nd("i_bp_lsf_nd");
  i_bp_lsf_nd->inp(s_in);
  i_bp_lsf_nd->outp(s_lsf_nd_out);

  ////////////////////////////////////////////////

  band_lsf_zp* i_bp_lsf_zp = new band_lsf_zp("i_bp_lsf_zp");
  i_bp_lsf_zp->inp(s_in);
  i_bp_lsf_zp->outp(s_lsf_zp_out);

  ////////////////////////////////////////////////

  sca_util::sca_trace_file* tf = sca_util::sca_create_tabular_trace_file("band_pass_td.dat");
  sca_util::sca_trace(tf, s_in, "s_in");
  sca_util::sca_trace(tf, s_out_nd, "s_out_nd");
  sca_util::sca_trace(tf, s_out_zp, "s_out_zp");
  sca_util::sca_trace(tf, n_out_rc, "n_out_rc"); // voltage at node
  sca_util::sca_trace(tf, s_lsf_nd_out, "s_lsf_nd_out");
  sca_util::sca_trace(tf, s_lsf_zp_out, "s_lsf_zp_out");
  sca_util::sca_trace(tf, i_bp_rc->n_in, "i_bp_rc.n_in"); // voltage of internal node
  sca_util::sca_trace(tf, i_bp_rc->r1, "i_bp_rc.r1"); // current through r1
  sca_util::sca_trace(tf, i_bp_rc->c1, "i_bp_rc.c1"); // current through c1

  ////////////////////////////////////////////////

  sc_core::sc_start(5.0, sc_core::SC_MS); // start time domain simulation for 5ms

  std::cout << sc_core::sc_time_stamp() << " time domain simulation finished" << std::endl
            << std::endl;

  ////////////////////////////////////////////////

  // we store the ac result in another file
  tf->reopen("band_pass_ac.dat");
  // we store the ac result as magnitude in dB and phase in degree
  // instead the default complex numbers
  tf->set_mode(sca_util::sca_ac_format(sca_util::SCA_AC_DB_DEG));

  // start ac-simulation from 100Hz to 10kHz with 200 points logarithmically spaced
  sca_ac_analysis::sca_ac_start(10.0, 100.0e3, 400, sca_ac_analysis::SCA_LOG);

  sca_util::sca_close_tabular_trace_file(tf);

  std::cout << "AC-simulation finished" << std::endl;

  /////////////////////////////////////////////////////////////////////////
  //// Result check
  ////////////////////////////////////////////////////////////////////////

  std::cout << std::endl << "---------------------------------------------------------";
  std::cout << std::endl << "Check results of time domain simulation ..." << std::endl;

  test_util::check_results("band_pass_td", 9, "bandpass");

  std::cout << std::endl << "---------------------------------------------------------";
  std::cout << std::endl << "Check results of ac domain simulation ..." << std::endl;

  test_util::check_results("band_pass_ac", 18, "bandpass", 1e-12);

  delete i_src, i_bp_nd, i_bp_zp, i_bp_rc, i_bp_lsf_nd, i_bp_lsf_zp;

  TEST_LABEL_END;

  return 0;
}
