/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

//  circuit10.cpp  - eln regression test

#include <systemc-ams>
#include "test_utilities.h"

SC_MODULE(circuit10)
{  
  // port declaration
  sca_tdf::sca_in<double> inp1;

  // node declaration
  sca_eln::sca_node n1;
  sca_eln::sca_node_ref n2; // reference node
  sca_eln::sca_node n3;
  sca_eln::sca_node n4;
  sca_eln::sca_node n5;

  // component declaration
  sca_eln::sca_tdf_vsource *vin;
  sca_eln::sca_vsource     *vs;
  sca_eln::sca_r           *r1, *r2, *r3, *r4;

  struct params // parameter
  {
    double r1_val;
    double r2_val;
    double r3_val;
    double r4_val;

    params() // default for parameter
    {
      r1_val = 1.0e3;
      r2_val = 4.0e3;
      r3_val = 2.0e3;
      r4_val = 3.0e3;
    }
  };

  circuit10( sc_core::sc_module_name nm, params pa = params() )
  : inp1("inp1"), 
    n1("n1"), n2("n2"), n3("n3"), n4("n4"), n5("n5"),
    p(pa)
  {
    architecture(); // generate netlist
  }

  ~circuit10();

 private:
  params p;
  void architecture(); // method for netlist
};

// netlist implementation

void circuit10::architecture()
{
  r1 = new sca_eln::sca_r("r1");
  r1->value = p.r1_val;
  r1->n(n1);
  r1->p(n5);

  r2 = new sca_eln::sca_r("r2");
  r2->value = p.r2_val;
  r2->n(n2);
  r2->p(n1);

  r3 = new sca_eln::sca_r("r3");
  r3->value = p.r3_val;
  r3->n(n3);
  r3->p(n2);

  r4 = new sca_eln::sca_r("r4");
  r4->value = p.r4_val;
  r4->n(n4);
  r4->p(n3);

  vs = new sca_eln::sca_vsource("vs");
  vs->p(n1);
  vs->n(n3);
  vs->frequency   = 51;
  vs->amplitude   = 12;
  vs->offset      = 2;
  vs->init_value  = 10.0;
  vs->delay       = sc_core::sc_time(0.005, sc_core::SC_SEC);
  vs->ac_amplitude= 0.0;
  vs->ac_phase    = 0.0;

  vin = new sca_eln::sca_tdf_vsource("vin");
  vin->inp(inp1);
  vin->p(n5);
  vin->n(n4);
}

circuit10::~circuit10()
{
  delete r1, r2, r3, r4, vs, vin;
}

void check_td_results(std::string fname)
{
  bool test_passed_with_success = true;
  std::ifstream tdf(fname.c_str());
  std::string line;
  long test_cnt = 0;

  getline(tdf, line);
  std::cout << "1..5" << std::endl;
  while (getline(tdf, line))
  {
    std::istringstream str(line);

    double time;
    str >> time;

    double node1, node3;
    str >> node1 >> node3;

    if (time <= 0.005)
    {
      if (!test_util::check_value(node1 - node3, 10.0))
      {
        std::cerr << "Error TD-check failed for verification of the output value of the sinusoidal voltage source at: " << time << std::endl;
        std::cout << "not ";
        test_passed_with_success = false;
      }
    }
    else
    {
      if (!test_util::check_value(node1 - node3, 2 + (12 * std::sin(2.0 * M_PI * 51 * (time - 0.005)))))
      {
        std::cerr << "Error TD-check failed for verification of the output value of the sinusoidal voltage source at: " << time << std::endl;
        std::cout << "not ";
        test_passed_with_success = false;
      }
    }
    std::cout << "ok " << ++test_cnt << std::endl;
  };

  if (test_passed_with_success) std::cout  << "results ok." << std::endl;
}

int sc_main(int argc, char* argv[])
{
  TEST_LABEL_START;

  // define non-conservative signals
  sca_tdf::sca_signal<double> signal1;

  test_util::const_src::params src1_params;
  src1_params.ampl = 10;

  test_util::const_src* src1 = new test_util::const_src("source1", src1_params);
  src1->outp(signal1);

  circuit10* cir = new circuit10 ("circuit");
  cir->inp1(signal1);

  sca_util::sca_trace_file* tf = sca_util::sca_create_tabular_trace_file("output10.dat");
  sca_util::sca_trace(tf, cir-> n1, "node1");
  sca_util::sca_trace(tf, cir-> n3, "node3");

  sc_core::sc_start(50, sc_core::SC_MS);

  sca_util::sca_close_tabular_trace_file(tf);

  check_td_results("output10.dat");

  TEST_LABEL_END;

  delete src1, cir;

  return 0;
}
