/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/
// test01.cpp  - eln regression test
// test01 with one de-capacitor and others

#include <systemc-ams>
#include "test_utilities.h"

///////////////////////////////////////////////////////////////////////////////

SC_MODULE(test01)
{
  // port declaration
  sca_tdf::sca_in<double> inp1;
  sca_tdf::sca_signal<double> sig1;
  sc_core::sc_signal<double> sig2;

  // node declaration
  sca_eln::sca_node n1;
  sca_eln::sca_node n2;
  sca_eln::sca_node_ref gnd; // reference node

  struct params // parameter
  {
    params() // default for parameter
    {
    }

  };

  // component declaration
  sca_eln::sca_tdf_isource *iin;
  sca_eln::sca_de_c        *de_c1;
  sca_eln::sca_r           *r1;
  test_util::conv_tdf2de   *conv_tdf2de1;

  test01( sc_core::sc_module_name nm, params pa = params() )
  : inp1("inp1"), sig1("sig1"), sig2("sig2"), 
    n1("n1"), n2("n2"), gnd("gnd"), p(pa)
  {
    sig2.write(1.0);
    architecture(); // generate netlist
  }

  ~test01();

private:
  params p;
  void architecture(); // method for netlist
};

// netlist implementation

void test01::architecture()
{
  conv_tdf2de1 = new test_util::conv_tdf2de("conf_tdf2de1");
  conv_tdf2de1->sc_o(sig2);
  conv_tdf2de1->tdf_i(sig1);

  iin = new sca_eln::sca_tdf_isource("vin");
  iin->inp(inp1);
  iin->p(n1);
  iin->n(gnd);

  de_c1 = new sca_eln::sca_de_c("de_c1");
  de_c1->scale = 1.0e-3;
  de_c1->n(n1);
  de_c1->p(n2);
  de_c1->inp(sig2);

  r1 = new sca_eln::sca_r("r1");
  r1->value = 1.0;
  r1->n(n2);
  r1->p(gnd);
}

test01::~test01()
{
  delete conv_tdf2de1;
  delete iin, de_c1, r1;
}

void check_td_results(std::string fname)
{
  bool test_passed_with_success = true;
  std::ifstream tdf(fname.c_str());
  std::string line;
  long test_cnt = 0;

  getline(tdf, line);

  double first_integral(0.0), second_integral(0.0);

  std::cout << "1..10" << std::endl;
  while (getline(tdf, line))
  {
    std::istringstream str(line);

    double time;
    str >> time;

    double i_dec, c_val, v_r1, v_cde;
    str >> i_dec >> c_val >> v_r1 >> v_cde;

    if ( !(test_util::check_value(i_dec, 10.0) && test_util::check_value(v_r1, -10.0)))
    {
      std::cerr << "Error TD-check failed for verification of currents at: " << time << std::endl;
      std::cout << "not ";
      test_passed_with_success = false;
    }
    std::cout << "ok " << ++test_cnt << std::endl;


    if (time < 0.05)
    {
      double expected_value = -time * 10.0 / (1e-3) - 10.0;
      if ( !(test_util::check_value(v_cde, expected_value   )))
      {
        std::cerr << "Error TD-check failed for v_cde at: " << time
            << " expect: " << expected_value << " got: " << v_cde<< std::endl;
        std::cout << "not ";
        test_passed_with_success = false;
      }
      std::cout << "ok " << ++test_cnt << std::endl;

      first_integral = expected_value- (-10.0);
    }

    if ((time >= 0.05) && (time < 0.09))
    {
      double expected_value = -(time - 0.04) * 10.0 / (1e-3 * 1e-12) - 10.0 + 
                              first_integral * 1e-3 / (1e-12 * 1e-3);

      if ( !(test_util::check_value(v_cde, expected_value    )))
      {
        std::cerr << "Error TD-check failed for v_cde at: " << time
            << " expect: " << expected_value << " got: " << v_cde<< std::endl;
        std::cout << "not ";
        test_passed_with_success = false;
      }
      std::cout << "ok " << ++test_cnt << std::endl;

      second_integral = expected_value - (-10.0);
    }

    if (time >= 0.09)
    {
      double expected_value = -(time - 0.08) * 10.0 / (1e-3 * 1e12) - 10.0 + 
                              second_integral * 1e-12 * 1e-3 / (1e12 * 1e-3);

      if ( !(test_util::check_value(v_cde, expected_value)) )
      {
        std::cerr << "Error TD-check failed for v_cde at: " << time
            << " expect: " << expected_value << " got: " << v_cde<< std::endl;
        std::cout << "not ";
        test_passed_with_success = false;
      }
      std::cout << "ok " << ++test_cnt << std::endl;
    }
  };
  if (test_passed_with_success) std::cout  << "results ok." << std::endl;
}

int sc_main(int argc, char* argv[])
{
  TEST_LABEL_START;

  // define non-conservative signals
  sca_tdf::sca_signal<double> signal1;

  test_util::const_src::params src_params;
  src_params.ampl = 10;

  test_util::const_src* src = new test_util::const_src("source", src_params);
  src->outp(signal1);

  test01* cir = new test01("test01");
  cir->inp1(signal1);

  test_util::pulse_src2::params src_params2;
  src_params2.val = 1.0;

  test_util::pulse_src2* tdf_src = new test_util::pulse_src2("tdf_source", src_params2);
  tdf_src->outp(cir->sig1);

  sca_util::sca_trace_file* tf = sca_util::sca_create_tabular_trace_file("output4.dat");
  sca_util::sca_trace(tf, cir->de_c1, "i_dec");
  sca_util::sca_trace(tf, cir->sig1, "c_val");
  sca_util::sca_trace(tf, cir->n2, "v_r1");
  sca_util::sca_trace(tf, cir->n1, "v_cde");

  sc_core::sc_start(120, sc_core::SC_MS);

  sca_util::sca_close_tabular_trace_file(tf);

  check_td_results("output4.dat");

  TEST_LABEL_END;

  delete src, cir, tdf_src;
  
  return 0;
}
