/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

#include <systemc>
#include <systemc-ams>
#include "test_utilities.h"

// =============================================================
//
//                                         [is 2V]
//                                          vsup
//  [is 1V]            [should be 1V]      +----+
//    vin                  vout            |    |
//   +--------ncp   np------+              p    |
//   |         (CCVS)       |            (R=1k) |
//   p      +-ncn   nn-+    |   [is 1V]    n    |
// (I=0)    |          |    |     vcm      |    p
//   n      +----------*----|--------------*  (V=2)
//   |                 |    |              |    n
//   +-----------------+    |              |    |
//                          p              p    |
//      ----------outp<(DE_SINK)         (R=1k) |
//      testval             n              p    |
//      [<0.5? bad]         |              |    |
//      [>=0.5? ok]         +--------------*----+
//                                         |
//  ### ... fix by adding 0Ohm            ---
//          resistor here
//
// =============================================================


// the DUT
SC_MODULE(ccvs_example)
{
  //instances
  sca_eln::sca_vsource i_vsup;
  sca_eln::sca_isource i_iin;
  sca_eln::sca_r i_sca_r1;
  sca_eln::sca_r i_sca_r2;
  sca_eln::sca_ccvs i_sca_ccvs;
  sca_eln::sca_de_vsink i_vout;

  SC_CTOR(ccvs_example)
  : i_vsup("i_vsup", 2.0, 2.0), // 2V DC
    i_iin("i_iin", 0.0, 0.0),   // 0A DC
    i_sca_r1("i_sca_r1", 1e3),  // R-Divider
    i_sca_r2("i_sca_r2", 1e3),  // R-Divider
    i_sca_ccvs("i_sca_ccvs", 1.0), // gain=1
    i_vout("i_vout", 1.0),           // output voltage
    vin("vin"), vsup("vsup"), vout("vout"), vcm("vcm"),
    gnd("gnd"),
    testval("testval")
  {
    // supply source
    i_vsup.p(vsup);
    i_vsup.n(gnd);
    i_vsup.set_timestep(1, sc_core::SC_NS);
  
    // input source
    i_iin.p(vin);
    i_iin.n(vcm);
  
    // generate vcm potential at half supply
    i_sca_r1.p(vsup);
    i_sca_r1.n(vcm);
    i_sca_r2.p(vcm);
    i_sca_r2.n(gnd);
  
    // CCVS connected on input and output to vcm
    i_sca_ccvs.ncp(vin);
    i_sca_ccvs.ncn(vcm);
    i_sca_ccvs.np(vout);
    i_sca_ccvs.nn(vcm);

    // voltage sink to test correct result
    i_vout.p(vout);
    i_vout.n(gnd);
    i_vout.outp(testval);
  }

  //nodes
  sca_eln::sca_node vin;
  sca_eln::sca_node vsup;
  sca_eln::sca_node vout;
  sca_eln::sca_node vcm;
  sca_eln::sca_node_ref gnd;

  sc_core::sc_signal<double> testval;
};

// =============================================================
// SC MAIN as simple test bench
// =============================================================

int sc_main(int argc, char* argv[])
{
  using namespace sc_core;

  TEST_LABEL_START;

  // matches timestep setup in DUT
  sc_set_time_resolution(1.0, SC_NS);
  
  // this is the CCVS DUT example
  ccvs_example dut("dut");
  
  // just generate one output value
  sc_start(2.0, SC_NS);
   
  // simple check if result is OK or not...
  if (dut.testval.read() < 0.5)
    SC_REPORT_ERROR("sc_main", "output of example is wrong (should be 1V)");
  else
    SC_REPORT_INFO("sc_main", "output of example is ok (it is 1V)");

  TEST_LABEL_END;

  return 0;
}
