/**
 *
 * @file cost_function.cc
 * @author Lasse Lehtonen
 *
 *
 */

/*
 * Copyright 2010 Tampere University of Technology
 * 
 *  This file is part of Transaction Generator.
 *
 *  Transaction Generator is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Transaction Generator is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Transaction Generator.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * $Id: cost_function.cc 1399 2010-08-26 13:56:45Z lehton87 $
 *
 */


#include "cost_function.hh"

#include <boost/config/warning_disable.hpp>
#include <boost/spirit/include/qi.hpp>
#include <boost/spirit/include/phoenix_operator.hpp>
#include <boost/spirit/include/phoenix_object.hpp>

#include <iostream>
#include <string>


namespace sctg
{
   // Namespace to hide nasty globals
   namespace cost_parse
   {
      namespace qi = boost::spirit::qi;
      namespace phoenix = boost::phoenix;
      namespace ascii = boost::spirit::ascii;

      struct sym_ : boost::spirit::qi::symbols<char, double>
      {
	 sym_()
	 {
	    add
	       ("the_answer", 42.0);
	 }
      } sym;

      template <typename Iterator>
      struct CostParser : qi::grammar<Iterator, double(), ascii::space_type>
      {
	 CostParser() : CostParser::base_type(expression)
	 {	    
            using namespace qi::labels;

            using phoenix::construct;
            using phoenix::val;

            expression =
	       term              [_val  = boost::spirit::_1]
	       >> *(('+' > term  [_val += boost::spirit::_1])
		   |('-' > term  [_val -= boost::spirit::_1])
		  );

            term =
	       factor             [_val  = boost::spirit::_1]
	       >> *(('*' > factor [_val *= boost::spirit::_1])
		   |('/' > factor [_val /= boost::spirit::_1])
		  )
	       ;

            factor =
	       qi::double_        [_val =  boost::spirit::_1]
	       | '(' > expression [_val =  boost::spirit::_1] > ')'
	       |('-' > factor     [_val = -boost::spirit::_1])
	       |('+' > factor     [_val =  boost::spirit::_1])
	       | sym              [_val =  boost::spirit::_1]
	       ;

	    qi::on_error<qi::fail>
	       (
		  expression
		  , std::cout
		  << val("Cost function parser failed! Did not understand \"")
		  << construct<std::string>(boost::spirit::_3, 
					    boost::spirit::_2)
		  << val("\"")
		  << std::endl
		  );

	 }

	 qi::rule<Iterator, double(), ascii::space_type> 
	 expression, term, factor;
	 
      };

   }

      
   CostFunction::CostFunction()
   {
   }

   CostFunction::~CostFunction()
   {
   }

   void CostFunction::addVariable(std::string& str, double value)
   {
      cost_parse::sym.add(str.c_str(), value);
   }

   double CostFunction::parse(std::string& function)
   {
      using boost::spirit::ascii::space;
      typedef std::string::const_iterator iterator_type;
      typedef cost_parse::CostParser<iterator_type> CostParser;

      CostParser costParser;
      double result = 0.0;

      std::string::const_iterator begin = function.begin();
      std::string::const_iterator end   = function.end();

      bool ok = phrase_parse(begin, end, costParser, space, result);
	 
      if(ok)
      {
	 return result;
      }
      else
      {
	 std::cout << "Cost function \"" << function
		   << "\" failed " << std::endl;
	 return 0.0;
      }
   }

}

// Local Variables:
// mode: c++
// c-file-style: "ellemtel"
// c-basic-offset: 3
// End:
