/*
 *
 * Copyright 2009 Tampere University of Technology
 * 
 *  This file is part of Execution Monitor.
 *
 *  Execution Monitor is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Execution Monitor is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Execution Monitor.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

package fi.cpu.ui;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.geom.Rectangle2D;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.ResourceBundle;
import java.util.Vector;

import javax.swing.BorderFactory;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.ScrollPaneConstants;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.DefaultTableModel;
import javax.swing.table.TableColumn;

import fi.cpu.Settings;
import fi.cpu.data.Chart;
import fi.cpu.data.Configuration;
import fi.cpu.data.ModelNode;
import fi.cpu.data.Process;
import fi.cpu.data.ProcessingElement;
import fi.cpu.data.Service;
import fi.cpu.data.Thread;
import fi.cpu.event.ModelEvent;
import fi.cpu.event.ModelListener;
import fi.cpu.event.ModelNodeEvent;
import fi.cpu.event.ModelNodeListener;
import fi.cpu.table.ReportTableCellRenderer;
import fi.cpu.table.ReportTableHeaderRenderer;
import fi.cpu.table.SortableTableModel;
import fi.cpu.ui.graph.GraphPanel;
import fi.cpu.ui.graph.SimpleChartFactory;

public class ProcessesPanel extends JPanel {
    private static final Color TEXT_BACKGROUND;
    private static final Color TEXT_COLOR;
    
    protected static Vector<String> COLUMN_NAMES = new Vector<String>();
    protected static final int FIRST_DATA_COLUMN;
    protected static final int FRACTION_DIGITS;
    
	protected Configuration config;
	protected MyProcessListener processListener;
	protected Vector<Process> processes;
	protected Process changedProcess;
	protected Map<Integer, ProcessingElement> processMapping;
	protected Map<Integer, Thread> threadMapping;
	protected Map<Integer, Service> serviceMapping;
	protected Map<Integer, Integer> signalsIn;
	protected Map<Integer, Integer> signalsOut;
    protected Font defaultFont;
	protected DefaultTableModel tableModel;
	protected JTable table;
	protected JPopupMenu cellPopupMenu;
	protected JPopupMenu rowPopupMenu;
	protected JPopupMenu columnPopupMenu;
	protected int popupRow = 0;
	protected int popupColumn = 0;

	static {
		ResourceBundle bundle = MainWindow.bundle;
		COLUMN_NAMES.add(bundle.getString("REPORT_LABEL_NAME"));
		COLUMN_NAMES.add(bundle.getString("REPORT_LABEL_SERVICE_MAP"));
		COLUMN_NAMES.add(bundle.getString("REPORT_LABEL_PE_MAP"));
		COLUMN_NAMES.add(bundle.getString("REPORT_LABEL_THREAD_MAP"));
		COLUMN_NAMES.add(bundle.getString("REPORT_LABEL_ECOUNT"));
		COLUMN_NAMES.add(bundle.getString("REPORT_LABEL_AVG_ETIME"));
		COLUMN_NAMES.add(bundle.getString("REPORT_LABEL_TOT_ETIME"));
		COLUMN_NAMES.add(bundle.getString("REPORT_LABEL_SIG_QUEUE"));
		COLUMN_NAMES.add(bundle.getString("REPORT_LABEL_LATENCY"));
		COLUMN_NAMES.add(bundle.getString("REPORT_LABEL_RESP_TIME"));
		COLUMN_NAMES.add(bundle.getString("REPORT_LABEL_SIG_IN_COUNT"));
		COLUMN_NAMES.add(bundle.getString("REPORT_LABEL_SIG_OUT_COUNT"));
		COLUMN_NAMES.add(bundle.getString("REPORT_LABEL_AVG_CTIME"));
		COLUMN_NAMES.add(bundle.getString("REPORT_LABEL_TOT_CTIME"));
		COLUMN_NAMES.add(bundle.getString("REPORT_LABEL_REL_CTIME"));
		COLUMN_NAMES.add(bundle.getString("REPORT_LABEL_LOCAL_CTIME"));
		COLUMN_NAMES.add(bundle.getString("REPORT_LABEL_LOCAL_CBYTES"));
		COLUMN_NAMES.add(bundle.getString("REPORT_LABEL_PENALTY"));
		COLUMN_NAMES.add(bundle.getString("REPORT_LABEL_REMOTE_CTIME"));
		COLUMN_NAMES.add(bundle.getString("REPORT_LABEL_REMOTE_CBYTES"));
		COLUMN_NAMES.add(bundle.getString("REPORT_LABEL_PENALTY"));
		
		FIRST_DATA_COLUMN = 4;
		
		FRACTION_DIGITS = Integer.parseInt(Settings.getAttributeValue("REPORT_FRACTION_DIGITS"));
		
        int red = 0;
        int green = 0;
        int blue = 0;
        
        red = Integer.parseInt( Settings.getAttributeValue("PROCESSOR_TEXT_COLOR_RED") );
        green = Integer.parseInt( Settings.getAttributeValue("PROCESSOR_TEXT_COLOR_GREEN") );
        blue = Integer.parseInt( Settings.getAttributeValue("PROCESSOR_TEXT_COLOR_BLUE") );
        TEXT_COLOR = new Color(red, green, blue);
 
        red = Integer.parseInt( Settings.getAttributeValue("PROCESSOR_TEXT_BACKGROUD_COLOR_RED") );
        green = Integer.parseInt( Settings.getAttributeValue("PROCESSOR_TEXT_BACKGROUD_COLOR_GREEN") );
        blue = Integer.parseInt( Settings.getAttributeValue("PROCESSOR_TEXT_BACKGROUD_COLOR_BLUE") );
        TEXT_BACKGROUND = new Color(red, green, blue);
	}
	
	/**
	 * Creates a new ComunicationPanel.
	 */
	public ProcessesPanel(Configuration config) {
		super();
		this.config = config;
		this.processListener = new MyProcessListener();
		processes = new Vector<Process>();
		processMapping = new HashMap<Integer, ProcessingElement>();
		threadMapping = new HashMap<Integer, Thread>();
		serviceMapping = new HashMap<Integer, Service>();
		signalsIn = new HashMap<Integer, Integer>();
		signalsOut = new HashMap<Integer, Integer>();
		
		this.config.getPeModel().addModelListener(new MyModelListener());
		this.config.getServiceModel().addModelListener(new MyModelListener());
		
        setBackground(TEXT_BACKGROUND);
        setBorder(BorderFactory.createLineBorder(Color.BLACK));
        setLayout(new GridBagLayout());

		defaultFont = getFont();
		tableModel = new SortableTableModel(COLUMN_NAMES, 0);
		
		// Create container for the table
		JPanel tableContainer = new JPanel();
		tableContainer.setLayout(new BorderLayout());

		// Create the table view
		table = new JTable(tableModel);
		table.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		table.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
		table.getTableHeader().setReorderingAllowed(false);
		table.setDefaultRenderer(Object.class, new ReportTableCellRenderer(defaultFont, defaultFont));
		table.setDefaultEditor(Object.class, null);
		table.setAutoCreateRowSorter(true);
		table.addMouseListener(new MyMouseListener());
		table.getTableHeader().addMouseListener(new MyMouseListener());
		
		// Set column widths to be large enough to show full titles
		TableColumn column = null;
		FontMetrics fm = null;
		Graphics g = getGraphics();
		for (int i = 0; i < COLUMN_NAMES.size(); ++i) {
			String columnTitle = null;

			columnTitle = COLUMN_NAMES.get(i);
			fm = getFontMetrics(defaultFont);
			
			Rectangle2D bounds = fm.getStringBounds(columnTitle, g);
			int width = (int)bounds.getWidth();

			column = table.getColumnModel().getColumn(i);
			column.setPreferredWidth(width+8); // a little bit extra for borders, empty space etc.
		}
		
		// Configure the table header
		DefaultTableCellRenderer headerRenderer = new ReportTableHeaderRenderer();
		table.getTableHeader().setDefaultRenderer(headerRenderer);

		// Add items to panel
		JScrollPane scrollPane = new JScrollPane(table, ScrollPaneConstants.VERTICAL_SCROLLBAR_AS_NEEDED,
		ScrollPaneConstants.HORIZONTAL_SCROLLBAR_AS_NEEDED);
        
        scrollPane.setBorder(null);
        scrollPane.getViewport().setBorder(null);
        this.add(scrollPane, new GridBagConstraints(
        		0, 0, GridBagConstraints.REMAINDER, GridBagConstraints.REMAINDER,
        		1.0, 1.0, GridBagConstraints.CENTER, GridBagConstraints.BOTH,
        		new Insets(0, 0, 0, 0), 0, 0));
        
        // Create popup menu for cell selection
		cellPopupMenu = new JPopupMenu();
		JMenuItem cellPopupMenuItem = new JMenuItem(MainWindow.bundle.getString("SHOW_GRAPH"));
		cellPopupMenuItem.addActionListener(new MyCellActionListener());
		cellPopupMenu.add(cellPopupMenuItem);
		
		// Create popup menu for row selection
		rowPopupMenu = new JPopupMenu();
		JMenuItem rowPopupMenuItem = new JMenuItem(MainWindow.bundle.getString("SHOW_GRAPHS_IN_ROW"));
		rowPopupMenuItem.addActionListener(new MyRowActionListener());
		rowPopupMenu.add(rowPopupMenuItem);
		
		// Create popup menu for column selection
		columnPopupMenu = new JPopupMenu();
		JMenuItem columnPopupMenuItem = new JMenuItem(MainWindow.bundle.getString("SHOW_GRAPHS_IN_COLUMN"));
		columnPopupMenuItem.addActionListener(new MyColumnActionListener());
		columnPopupMenu.add(columnPopupMenuItem);

		initPanel();
	}
	
	/**
	 * Initializes the panel.
	 */
	public void initPanel() {
		for (Process p : processes) {
			p.removeModelNodeListener(processListener);
		}
		processes.clear();
		
		List<ModelNode> processNodes = config.getPeModel().getNodesOfType(Process.class);
		Iterator<ModelNode> nodeIter = processNodes.iterator();

        while (nodeIter.hasNext()) {
        	ModelNode node = nodeIter.next();
        	if (!(node instanceof Process)) {
        		continue;
        	}
        	Process p = (Process) node;
        	processes.add(p);
        	processMapping.put(p.getId(), null);
        	threadMapping.put(p.getId(), null);
        	serviceMapping.put(p.getId(), null);
        	signalsIn.put(p.getId(), new Integer(0));
			signalsOut.put(p.getId(), new Integer(0));
        }
        
        // Sort processes
		Collections.sort(processes);
		
		// clear the table
		tableModel.setRowCount(0);
		
		// add process rows
		for (int i=0; i<processes.size(); ++i) {
			Process p = processes.get(i);
			
			Object[] rowData = makeProcessRow(p);
			
			tableModel.addRow(rowData);
			p.addModelNodeListener(processListener);
		}

		updateDataTable();
		updateMappingValues();
	}
	
	/**
	 * Adds signal count to source and destination processes.
	 */
	public void addSignalCount(int source, int destination, int count) {
		// Sum new signal value to old values
		Integer old = null;
		
		old = signalsIn.get(destination);
		if (old == null) {
			signalsIn.put(new Integer(destination), new Integer(count));
		} else {
			signalsIn.put(new Integer(destination), new Integer(count + old));
		}
		
		old = signalsOut.get(source);
		if (old == null) {
			signalsOut.put(new Integer(source), new Integer(count));
		} else {
			signalsOut.put(new Integer(source), new Integer(count + old));
		}
		
		sync();
	}
	
	/**
	 * Synchronizes view updating.
	 */
	public void sync() {
		for (int i = 0; i < processes.size(); ++i) {
			Process p = processes.get(i);
			int id = p.getId();
			
			// Update signal counts to process objects
			p.setSignalsInCount(signalsIn.get(id));
			p.setSignalsOutCount(signalsOut.get(id));
			
			// Clear signal counts for next measurement round
			signalsIn.put(id, new Integer(0));
			signalsOut.put(id, new Integer(0));
		}
		
		updateDataTable();
	}
	
	/**
	 * Makes new data row of a process for the processes table.
	 */
	protected Object[] makeProcessRow(Process p) {
		ProcessingElement pe = processMapping.get(p.getId());
		String peName = "";
		if (pe != null) {
			peName = pe.getName();
		}
		
		Thread thread = threadMapping.get(p.getId());
		String threadName = "";
		if (pe != null) {
			threadName = thread.getName();
		}
		
		Service service = serviceMapping.get(p.getId());
		String serviceName = "";
		if (service != null) {
			serviceName = service.getName();
		}
		
		Object[] rowData = {p.getName(),
				serviceName,
				peName,
				threadName,
				p.getExecCount(),
				p.getAvgExecTime(),
				p.getTotExecTime(),
				p.getSigQueue(),
				p.getLatency(),
				p.getResponseTime(),
				p.getSignalsInCount(),
				p.getSignalsOutCount(),
				p.getAvgCommTime(),
				p.getTotCommTime(),
				new Double(p.getRelCommTime()),
				p.getLocalCommTime(),
				p.getLocalCommBytes(),
				p.getLocalCommPenalty(),
				p.getRemoteCommTime(),
				p.getRemoteCommBytes(),
				p.getRemoteCommPenalty()};
		
		return rowData;
	}
	
	/**
	 * Updates the data in the table.
	 */
	protected void updateDataTable() {	
		// update process rows
		for (int i = 0; i < processes.size(); ++i) {			
			Process p = processes.get(i);
			
			Object[] rowData = makeProcessRow(p);
			
			for (int j = 0; j < rowData.length; ++j) {
				tableModel.setValueAt(rowData[j], i, j);
			}
			
			// update process charts
			for (int j = FIRST_DATA_COLUMN; j < table.getColumnCount(); ++j) {
				StringBuilder chartID = new StringBuilder();
				chartID.append(p.getName());
				chartID.append(":");
				chartID.append(table.getColumnName(j));
				
				Number value = (Number)table.getValueAt(i, j);
				MainWindow.getInstance().addValueToChart(
						null, chartID.toString(), value.doubleValue());
			}
		}
		
		// refresh the view
		revalidate();
		repaint();
	}
	
	/**
	 * Updates the column with mapping info
	 */
	protected void updateMappingValues() {
		List<ModelNode> pList = null;
		
        // Update PE & thread mapping info
		pList = config.getPeModel().getNodesOfType(Process.class);
        for (int i = 0; i < pList.size(); ++i) {
        	Process p = (Process) pList.get(i);
        	if (processMapping.containsKey(p.getId())) {
        		ModelNode peNode = p.getParentNode().getParentNode();
        		if (peNode instanceof ProcessingElement) {
        			ProcessingElement pe = (ProcessingElement) peNode;
        			processMapping.put(p.getId(), pe);
        		}
        	}
        	if (threadMapping.containsKey(p.getId())) {
        		ModelNode threadNode = p.getParentNode();
        		if (threadNode instanceof Thread) {
        			Thread thread = (Thread) threadNode;
        			threadMapping.put(p.getId(), thread);
        		}
        	}
        }
        
        // Update service mapping info
		pList = config.getServiceModel().getNodesOfType(Process.class);
        for (int i = 0; i < pList.size(); ++i) {
        	Process p = (Process) pList.get(i);
        	if (serviceMapping.containsKey(p.getId())) {
        		ModelNode serviceNode = p.getParentNode();
                if (serviceNode instanceof Service) {
                	Service service = (Service) serviceNode;
        			serviceMapping.put(p.getId(), service);
                }
        	}
        }
        
		for (int i = 0; i < processes.size(); ++i) {			
			Process p = processes.get(i);

			ProcessingElement pe = processMapping.get(p.getId());
			String peName = "";
			if (pe != null) {
				peName = pe.getName();
			}
			
			Thread thread = threadMapping.get(p.getId());
			String threadName = "";
			if (pe != null) {
				threadName = thread.getName();
			}
			
			Service service = serviceMapping.get(p.getId());
			String serviceName = "";
			if (service != null) {
				serviceName = service.getName();
			}
			
			tableModel.setValueAt(serviceName, i, 1);
			tableModel.setValueAt(peName, i, 2);
			tableModel.setValueAt(threadName, i, 3);
		}
	}
	
    /**
     * Listener for ModelEvents.
     */
    private class MyModelListener implements ModelListener {
    	public void nodeInserted(ModelEvent e) {
    		initPanel();
    	}
    	public void nodeRemoved(ModelEvent e) {
    		initPanel();
    	}
    	public void nodeMoved(ModelEvent e) {
    		initPanel();    			
    	}
    	public void structureChanged(ModelEvent e) {
    		initPanel();
    	}
    }
    
	/**
	 * Listener for MouseEvents to post the popup menu
	 */
	private class MyMouseListener extends MouseAdapter {
		public void mousePressed(MouseEvent e) {
			showPopupIfTriggered(e);
		}
		public void mouseReleased(MouseEvent e) {
			showPopupIfTriggered(e);
		}
		private void showPopupIfTriggered(MouseEvent e) {
			if (e.isPopupTrigger()) {
				Point p = e.getPoint();
				popupRow = table.rowAtPoint(p);
				popupColumn = table.columnAtPoint(p);
				
				// Is event from TableHeader or Table?
				if (e.getComponent() == table.getTableHeader()) {
					if (popupColumn >= FIRST_DATA_COLUMN) {
						columnPopupMenu.show(e.getComponent(), e.getX(), e.getY());
					}
					
				} else {
					if (popupColumn == 0) {
						rowPopupMenu.show(e.getComponent(), e.getX(), e.getY());
					} else if (popupColumn >= FIRST_DATA_COLUMN && popupRow >= 0) {
						cellPopupMenu.show(e.getComponent(), e.getX(), e.getY());
					}
				}
			}
		}
	}

	
	/**
	 * Listener for actions from the popup menu when sigle cell is selected.
	 */
	private class MyCellActionListener implements ActionListener {
		public void actionPerformed(ActionEvent e) {
			String process = (String)table.getValueAt(popupRow, 0);
			String variable = table.getColumnName(popupColumn);
			String graphID = process +":" + variable;
			String title = process + ": " + variable;
			int historySize = 0;
			
			try {
				String history = System.getProperty("history");
				if (history != null) {
					historySize = Integer.parseInt(history);
				}
			} catch (NumberFormatException nfe) {
				// do nothing
			}
			
			GraphPanel gPanel = new GraphPanel();
//			Chart chart = new Chart(graphID, SimpleChartFactory.LINECHART_TYPE,
//					title, MainWindow.bundle.getString("XAXIS"), "", 0.0, 0.0,
//					historySize, true, false);
			Chart chart = new Chart(graphID, SimpleChartFactory.LINECHART_TYPE,
					title, MainWindow.bundle.getString("XAXIS"), "", 0.0, 0.0,
					50, true, false);
			gPanel.setChart(chart);
			MainWindow.getInstance().addChart( gPanel, 10, 10, 200, 300 );
		}
	}
	
	/**
	 * Listener for actions from the popup menu when entire row is selected.
	 */
	private class MyRowActionListener implements ActionListener {
		public void actionPerformed(ActionEvent e) {		
			for (int i = FIRST_DATA_COLUMN; i < table.getColumnCount(); i++) {
				String process = (String)table.getValueAt(popupRow, 0);
				String variable = table.getColumnName(i);
				String graphID = process +":" + variable;
				String title = process + ": " + variable;
				int historySize = 0;
				
				try {
					String history = System.getProperty("history");
					if (history != null) {
						historySize = Integer.parseInt(history);
					}
				} catch (NumberFormatException nfe) {
					// do nothing
				}
				
				GraphPanel gPanel = new GraphPanel();
				Chart chart = new Chart(graphID, SimpleChartFactory.LINECHART_TYPE,
						title, MainWindow.bundle.getString("XAXIS"), "", 0.0, 0.0,
						historySize, true, false);
				gPanel.setChart(chart);
				MainWindow.getInstance().addChart( gPanel, 20 * i, 20 * i, 200, 300 );
			}
		}
	}
	
	/**
	 * Listener for actions from the popup menu when entire column is selected.
	 */
	private class MyColumnActionListener implements ActionListener {
		public void actionPerformed(ActionEvent e) {
			for (int i = 0; i < table.getRowCount(); i++) {
				String process = (String)table.getValueAt(i, 0);
				String variable = table.getColumnName(popupColumn);
				String graphID = process +":" + variable;
				String title = process + ": " + variable;
				int historySize = 0;
				
				try {
					String history = System.getProperty("history");
					if (history != null) {
						historySize = Integer.parseInt(history);
					}
				} catch (NumberFormatException nfe) {
					// do nothing
				}
				
				GraphPanel gPanel = new GraphPanel();
				Chart chart = new Chart(graphID, SimpleChartFactory.LINECHART_TYPE,
						title, MainWindow.bundle.getString("XAXIS"), "", 0.0, 0.0,
						historySize, true, false);
				gPanel.setChart(chart);
				MainWindow.getInstance().addChart( gPanel, 20 * i, 20 * i, 200, 300 );
			}
		}
	}
	
	/**
	 * Listener for events from contained processes.
	 */
	private class MyProcessListener implements ModelNodeListener {
		public void modelNodeChanged(ModelNodeEvent e) {
			if (e.getEventId() == Process.DATA_CHANGED_EVENT) {
				Object source = e.getSource();
				
				if (source instanceof Process) {
					Process p = (Process) source;
					
//					int index = processes.indexOf(p);
//					if (index == -1) {
//						return;
//					}
//					
//					Object[] rowData = makeProcessRow(p);
//					for (int i=0; i<rowData.length; ++i) {
//						tableModel.setValueAt(rowData[i], index, i);
//					}
					
					updateDataTable();
				}
			}
		}
	}
}
