/**
 * File:    Model.java
 * Author:  Tomi Jantti <tomi.jantti@tut.fi>
 * Created: 29.3.2007
 *
 *
 *
 * Copyright 2009 Tampere University of Technology
 * 
 *  This file is part of Execution Monitor.
 *
 *  Execution Monitor is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Execution Monitor is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Execution Monitor.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 */
package fi.cpu.data;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreeNode;

import fi.cpu.event.ModelEvent;
import fi.cpu.event.ModelListener;


/**
 * Model is a hierarchical structure of ModelNodes.
 */
public class Model extends DefaultTreeModel {
	private enum NotificationType {NODE_INSERTED, NODE_REMOVED, NODE_MOVED, STRUCTURE_CHANGED}
	private boolean eventsEnabled;
	
	/**
	 * Creates a new Model.
	 */
	public Model() {
		super(new ModelNode("root"));
		eventsEnabled = true;
	}
	
	
	/**
	 * @return Returns the modelRoot.
	 */
	public ModelNode getModelRoot() {
		return (ModelNode)getRoot();
	}
	
	
	/**
	 * @return Returns a list of ModelNodes that are instances of
	 * 		   the given class or one of its possible sub classes.
	 */
	public List<ModelNode> getNodesOfType(Class<? extends ModelNode> c) {
		List<ModelNode> foundNodes = new ArrayList<ModelNode>();
		addNodesOfType(c, getModelRoot(), foundNodes);
		return foundNodes;
	}
	
	
	/**
	 * Handles the currentNode and its descendants recursively,
	 * searching for nodes of given type.
	 */
	private void addNodesOfType(Class<? extends ModelNode> c,
			ModelNode currentNode, List<ModelNode> nodes) {
		if (c.isInstance(currentNode)) {
			nodes.add(currentNode);
		}
		
		// handle children recursively
		Iterator<ModelNode> childIter = getChildren(currentNode).iterator();
		while (childIter.hasNext()) {
			ModelNode child = childIter.next();
			addNodesOfType(c, child, nodes);
		}
	}
	
	
	/**
	 * @return Returns child nodes of the given node.
	 */
	public List<ModelNode> getChildren(ModelNode parent) {
		List<ModelNode> nodeList = new ArrayList<ModelNode>();
		
		for (int i=0; i<parent.getChildCount(); ++i) {
			nodeList.add((ModelNode)parent.getChildAt(i));
		}
		
		return nodeList;
	}

	
	/**
	 * @return Returns the number of child nodes the given node has.
	 */
	public int getChildCount(ModelNode parent) {
		return parent.getChildCount();
	}

	
	/**
	 * @return Returns the index'th child of the given node.
	 */
	public ModelNode getChildAt(ModelNode parent, int index) {
		return (ModelNode)parent.getChildAt(index);
	}
	
	
	/**
	 * @return Returns true if node1 is a child of node2.
	 */
	public boolean isChildOf(ModelNode node1, ModelNode node2) {
		return node2.isNodeChild(node1);
	}
	
	
	/**
	 * @return Returns a list of nodes forming a path
	 * from the root to the given node.
	 */
	public List<ModelNode> getPathFromRoot(ModelNode node) {
		TreeNode[] path = getPathToRoot(node);
		List<ModelNode> list = new ArrayList<ModelNode>();
		for (int i=0; i<path.length; ++i) {
			list.add((ModelNode)path[i]);
		}
		return list;
	}
	
	
	/**
	 * Inserts a node to the model.
	 * @param child  The node to be inserted.
	 * @param parent The parent of the new node.
	 */
	public void insertNode(ModelNode child, ModelNode parent) {
		insertNode(child, parent, getChildCount(parent));
	}

	
    /**
	 * Inserts a node to the model.
	 * @param child  The node to be inserted.
	 * @param parent The parent of the new node.
	 * @param index  The index to which insert.
	 */
	public void insertNode(ModelNode child, ModelNode parent, int index) {
		ModelNode oldParent = child.getParentNode();
		insertNodeInto(child, parent, index);
		
		if (oldParent != null) {
			oldParent.childRemoved(child);
		}
		parent.childAdded(child);
		
		fireNotification(NotificationType.NODE_INSERTED,
				new ModelEvent(this, child, parent, oldParent));
	}
	
	
	/**
	 * Removes a node from the model.
	 * @param node The node to be removed.
	 */
	public void removeNode(ModelNode node) {
		ModelNode oldParent = node.getParentNode();
		removeNodeFromParent(node);
		
		if (oldParent != null) {
			oldParent.childRemoved(node);
		}

		fireNotification(NotificationType.NODE_REMOVED,
				new ModelEvent(this, node, null, oldParent));
	}
	
	
	/**
	 * Removes all child nodes of a node.
	 * @param node
	 */
	public void removeAllChildren(ModelNode node) {
		node.removeAllChildren();
		fireNotification(NotificationType.STRUCTURE_CHANGED,
				new ModelEvent(this, node, null, null));
	}
	
	
	/**
	 * Moves a node to a new parent node.
	 * @param node      The node to be moved.
	 * @param newParent The node where to insert.
	 */
	public void moveNode(ModelNode node, ModelNode newParent) {
		moveNode(node, newParent, getChildCount(newParent));
	}

		
	/**
	 * Moves a node to a new parent node.
	 * @param node      The node to be moved.
	 * @param newParent The node where to insert.
	 * @param index     The index to which insert.
	 */
	public void moveNode(ModelNode node, ModelNode newParent, int index) {
		ModelNode oldParent = node.getParentNode();
		insertNodeInto(node, newParent, index);
		
		if (oldParent != null) {
			oldParent.childRemoved(node);
		}
		newParent.childAdded(node);

		fireNotification(NotificationType.NODE_MOVED,
				new ModelEvent(this, node, newParent, oldParent));		
	}
	
	
	/**
	 * Enables/disables ModelEvent notification.
	 */
	public void enableEvents(boolean enable) {
		eventsEnabled = enable;
	}
	
	
	/**
	 * This method should be called if the model has been modified
	 * with events disabled. This results in notification of
	 * structureChanged-event.
	 * @param node The node highest in the modified model hierarchy.
	 */
	public void modelStructureChanged(ModelNode node) {
		fireNotification(NotificationType.STRUCTURE_CHANGED,
				new ModelEvent(this, node, null, null));
	}
	
	
	/**
	 * Registers a listener to receive ModelEvents.
	 * @param l
	 */
	public void addModelListener(ModelListener l) {
		listenerList.add(ModelListener.class, l);
	}
	
	
	/**
	 * Unregisters a listener from receiving ModelEvents.
	 * @param l
	 */
	public void removeModelListener(ModelListener l) {
		listenerList.remove(ModelListener.class, l);
	}
	
	
	/**
	 * Fires a notification of event to all interested listeners.
	 */
	private void fireNotification(NotificationType type, ModelEvent e) {
		if (!eventsEnabled) {
			return;
		}
		
		Object[] listeners = listenerList.getListenerList();
		for (int i = listeners.length-2; i>=0; i-=2) {
			if (listeners[i] == ModelListener.class) {
				ModelListener listener = (ModelListener)listeners[i+1];
				
				switch (type) {
				case NODE_INSERTED:
					listener.nodeInserted(e);
					break;
				case NODE_REMOVED:
					listener.nodeRemoved(e);
					break;
				case NODE_MOVED:
					listener.nodeMoved(e);
					break;
				case STRUCTURE_CHANGED:
					listener.structureChanged(e);
					break;
				default:
					break;
				}
			}
		}
	}
}
