/*
  James Aldis
  TI France
  OCP TL1 Channel Timing Distribution
  29 June 2005

  C++ Source File for:
    top level of simulation testbench
*/


#include <string>
#include <map>

#include "tl1_timing_master.h"
#include "tl1_timing_slave.h"
#include "tl1_timing_merger.h"
#include "tl1_timing_splitter.h"


// forward declarations
OCP_TL1_CHANNEL *new_ocp_channel(
  string name, bool traces, int nr_threads, sc_clock &clk,
  OCP_MASTER_PORT &mport, OCP_SLAVE_PORT &sp);


// programme
int sc_main(int argc, char **argv)
{
  int rand_seed = 1;
  int traces = 1;

  double duration = 5000.0;
  double clk_period = 5.0;

  // **** SIMULATION CONFIGURATION ****
  cout << "usage: ./tl1_timing_test [d duration] [r rand-seed] [c clk-period] [t traces]" << endl;
  cout << "  using defaults for unspecified parameters" << endl;

  char **nextarg = argv;
  int argnr = 1;
  while((argnr+1) < argc) {
    nextarg++;
    argnr += 2; // always have a letter then a parameter
    switch(**nextarg) {
      case 'd':
        sscanf(*(++nextarg),"%lf",&duration);
        break;
      case 'r':
        sscanf(*(++nextarg),"%d",&rand_seed);
        break;
      case 'c':
        sscanf(*(++nextarg),"%lf",&clk_period);
        break;
      case 't':
        sscanf(*(++nextarg),"%d",&traces);
        break;
      default:
        cout << "unknown command-line parameter - " << **nextarg << " - trying to continue\n";
        argnr--;
    }
  }

  // simulator configuration
  sc_set_time_resolution(1.0,SC_PS);
  sc_set_default_time_unit(1.0,SC_NS);
  srandom(rand_seed);

  // **** MODULE INSTANTIATION ****
  // clock source
  sc_clock clk("clk",clk_period,0.5,0,true);

  // a collection of tl1_timing_master, tl1_timing_slave,
  // tl1_timing_splitter and tl1_timing_merger, making a (fully
  // combinatorial) network, with OCP-TL1-channels between them.
  // in this case we have a 4x4 shared bus.

  // masters are single-threaded
  tl1_timing_master mA("mA");
  tl1_timing_master mB("mB");
  tl1_timing_master mC("mC");
  tl1_timing_master mD("mD");

  // mergers have (N,M) threads on slave ports and N+M on master port
  tl1_timing_merger mgAB("mgAB", 1, 1);
  tl1_timing_merger mgCD("mgCD", 1, 1);
  tl1_timing_merger mgABCD("mgABCD", 2, 2);

  // splitters have N threads on slave port and (N,N) on master ports
  tl1_timing_splitter spABCD("spABCD", 4);
  tl1_timing_splitter spAB("spAB", 4);
  tl1_timing_splitter spCD("spCD", 4);

  // slaves are N-threaded
  tl1_timing_slave sA("sA", 4);
  tl1_timing_slave sB("sB", 4);
  tl1_timing_slave sC("sC", 4);
  tl1_timing_slave sD("sD", 4);

  // **** CHANNEL INSTANTIATION, CONFIGURATION AND BINDING ****
  // memory leak here as we discard the pointers to the channels
  // but not important

  // masters
  new_ocp_channel(string("mAocp"), traces, 1, clk, mA.ocp, mgAB.ocpsA);
  new_ocp_channel(string("mBocp"), traces, 1, clk, mB.ocp, mgAB.ocpsB);
  new_ocp_channel(string("mCocp"), traces, 1, clk, mC.ocp, mgCD.ocpsA);
  new_ocp_channel(string("mDocp"), traces, 1, clk, mD.ocp, mgCD.ocpsB);

  // after first level mergers
  new_ocp_channel(string("mgABocp"), traces, 2, clk, mgAB.ocpm, mgABCD.ocpsA);
  new_ocp_channel(string("mgCDocp"), traces, 2, clk, mgCD.ocpm, mgABCD.ocpsB);

  // after second level merger
  new_ocp_channel(string("mgABCDocp"), traces, 4, clk, mgABCD.ocpm, spABCD.ocps);

  // after first level splitters
  new_ocp_channel(string("spABocp"), traces, 4, clk, spABCD.ocpmA, spAB.ocps);
  new_ocp_channel(string("spCDocp"), traces, 4, clk, spABCD.ocpmB, spCD.ocps);

  // after second level splitters
  new_ocp_channel(string("sAocp"), traces, 4, clk, spAB.ocpmA, sA.ocp);
  new_ocp_channel(string("sBocp"), traces, 4, clk, spAB.ocpmB, sB.ocp);
  new_ocp_channel(string("sCocp"), traces, 4, clk, spCD.ocpmA, sC.ocp);
  new_ocp_channel(string("sDocp"), traces, 4, clk, spCD.ocpmB, sD.ocp);

  // **** OTHER BINDINGS ****
  mA.clk(clk);
  mB.clk(clk);
  mC.clk(clk);
  mD.clk(clk);
  sA.clk(clk);
  sB.clk(clk);
  sC.clk(clk);
  sD.clk(clk);
  mgAB.clk(clk);
  mgCD.clk(clk);
  mgABCD.clk(clk);
  spABCD.clk(clk);
  spAB.clk(clk);
  spCD.clk(clk);

  // **** SIMULATION ****
  cout << "Starting simulation" << endl;
  cout << "  duration = " << duration << endl;
  cout << "  rand seed = " << rand_seed << endl;
  cout << "  clk frequency = " << 1.0/clk_period << endl;
  cout << "  traces " << (bool(traces) ? "on" : "off") << endl;
  sc_start(duration);
  cout << "Simulation complete " << endl;

  return(0);
}



// configuration class creation for OCP channels

class MapStringCl : public std::map<std::string, std::string> {
  public:
    MapStringCl() : secondString(false), lhs("") {};

    MapStringCl & operator << (const string newstr) {
      if(secondString) {
        insert(std::make_pair(lhs,newstr));
        secondString = false;
      } else {
        lhs = newstr;
        secondString = true;
      }
      return(*this);
    };

  private:
    bool secondString;
    string lhs;
};


void configure_ocp(OCP_TL1_CHANNEL *module, int threads) {
  MapStringCl channel_config;

  channel_config <<   "addr_wdth"           <<   "i:32"  ;
  channel_config <<   "data_wdth"           <<   "i:32"  ;
  channel_config <<   "cmdaccept"           <<   "i:0"   ;
  channel_config <<   "respaccept"          <<   "i:0"   ;
  channel_config <<   "byte_enable"         <<   "i:1"   ;
  channel_config <<   "force_aligned"       <<   "i:1"   ;
  channel_config <<   "read_enable"         <<   "i:1"   ;
  channel_config <<   "write_enable"        <<   "i:1"   ;
  channel_config <<   "writenonpost_enable" <<   "i:1"   ;
  if(threads>9) {
    cout << "ERROR - too many threads\n";
    exit(1);
  }
  char threads_char = '0'+threads;
  string threads_string = string("i:") + threads_char;
  channel_config <<   "threads"             <<   threads_string   ;
  channel_config <<   "readex_enable"       <<   "i:1"   ;
  channel_config <<   "writeresp_enable"    <<   "i:1"   ;
  channel_config <<   "rdlwrc_enable"       <<   "i:0"   ;
  channel_config <<   "connid"              <<   "i:0"   ;
  channel_config <<   "mthreadbusy"         <<   "i:1"   ;
  channel_config <<   "sthreadbusy"         <<   "i:1"   ;
  channel_config <<   "mthreadbusyexact"    <<   "i:1"   ;
  channel_config <<   "sthreadbusyexact"    <<   "i:1"   ;
  channel_config <<   "burstseq"            <<   "i:0"   ;
  channel_config <<   "burstlength"         <<   "i:0"   ;
  channel_config <<   "burstlength_wdth"    <<   "i:1"   ;
  channel_config <<   "burstprecise"        <<   "i:0"   ;
  channel_config <<   "burst_aligned"       <<   "i:1"   ;
  channel_config <<   "reqlast"             <<   "i:0"   ;
  channel_config <<   "burstsinglereq"      <<   "i:0"   ;
  channel_config <<   "resplast"            <<   "i:0"   ;
  channel_config <<   "burstseq_incr_enable" <<  "i:1"   ;
  channel_config <<   "burstseq_dflt1_enable" << "i:0"   ;
  channel_config <<   "burstseq_dflt2_enable" << "i:0"   ;
  channel_config <<   "burstseq_strm_enable" <<  "i:0"   ;
  channel_config <<   "burstseq_unkn_enable" <<  "i:0"   ;
  channel_config <<   "burstseq_wrap_enable" <<  "i:0"   ;
  channel_config <<   "burstseq_xor_enable" <<   "i:0"   ;
  channel_config <<   "interrupt"           <<   "i:0"   ;
  channel_config <<   "merror"              <<   "i:0"   ;
  channel_config <<   "mreset"              <<   "i:0"   ;
  channel_config <<   "serror"              <<   "i:0"   ;
  channel_config <<   "sreset"              <<   "i:1"   ;

  module->setConfiguration(channel_config);
}


// helper function to instantiate, configure and bind an OCP TL1 channel
OCP_TL1_CHANNEL *new_ocp_channel(
  string name, bool traces, int nr_threads, sc_clock &clk,
  OCP_MASTER_PORT &mport, OCP_SLAVE_PORT &sport) {

  string f = (traces ? name + string(".ocp") : string(""));
  OCP_TL1_CHANNEL *tmp;
//   tmp = new OCP_TL1_CHANNEL(name, &clk, f);
  tmp = new OCP_TL1_CHANNEL(name.c_str(), f);
  tmp->p_clk(clk);
  configure_ocp(tmp, nr_threads);
  mport(*tmp);
  sport(*tmp);

  return(tmp);
}

