/*
  James Aldis
  TI France
  OCP TL1 Channel Timing Distribution
  29 June 2005

  C++ Header File for:
    single-threaded thread-busy-exact test master
*/


// multiple inclusion protection
#ifndef TL1_TIMING_MASTER_H
#define TL1_TIMING_MASTER_H


#include "tl1_timing_common.h"


// master derives from slave-timing-if, because it is
// single-ported and timing-sensitive.
class tl1_timing_master : 
  public sc_module, 
  public OCP_TL1_Slave_TimingIF 
{
  SC_HAS_PROCESS(tl1_timing_master);
  public:
    tl1_timing_master(sc_module_name name) :
      sc_module(name), ocp("ocp") {

      SC_THREAD(clock_rising);
      sensitive(clk.pos());
      dont_initialize();

      requests_sent = 0;
      responses_got = 0;
      requests_outstanding = 0;

      resp_expected = new int[max_requests_outstanding];
      rd_ptr = 0;
      wr_ptr = 0;

      time_quantum = sc_get_time_resolution();
      sthreadbusy_sample_time = time_quantum;  // initial guess
    };

    // destructor to print some stats
    ~tl1_timing_master() {
      cout << "Deleting master: " << name() << endl;
      cout << "  Requests sent: " << requests_sent << endl;
      cout << "  Responses got: " << responses_got << endl;
      cout << "  SThreadBusy sample time: " << sthreadbusy_sample_time << endl;
    };

    // ports
    sc_in<bool> clk;
    OCP_MASTER_PORT ocp;

    // processes
    void clock_rising() {
      while(true) {
        wait();
        // assert MThreadBusy directly after clock edge
        int mtb = (unit_rand() < pr_busy ? 1 : 0);
        ocp->putMThreadBusy(mtb);

        // get any response from previous cycle
        OCP_RESP resp;
        if(ocp->getOCPResponse(resp)) {
          requests_outstanding--;
          responses_got++;
          if((resp.SData & 0xFFFF) != resp_expected[rd_ptr]) {
            cout << name() << " " << sc_time_stamp() <<
              " ERROR: out of order response\n";
            cout << "expected: " << hex << resp_expected[rd_ptr] << dec << endl;
            cout << "got:      " << hex << resp.SData << dec << endl;
            sc_stop();
          }
          rd_ptr = (rd_ptr+1) % max_requests_outstanding;
        }

        // wait for long enough that SThreadBusy is stable
        wait(sthreadbusy_sample_time);
        int stb = (ocp->getSThreadBusy() & 1);

        // start a new request if we want to
        if((unit_rand() < pr_request) && (stb == 0) &&
           (requests_outstanding < max_requests_outstanding)) {
          OCP_REQ req;
          req.MCmd = OCP_MCMD_RD;
          req.MAddr = random();
          ocp->startOCPRequest(req);
          requests_outstanding++;
          requests_sent++;
          resp_expected[wr_ptr] = req.MAddr & 0xFFFF;
          wr_ptr = (wr_ptr+1) % max_requests_outstanding;
        }
      }
    };

    void end_of_elaboration() {
      cout << "<<<< E-O-E >>>> " << name() << endl;

      // master is not default-timing and therefore must inform the
      // channel of its actual timing
      // only non-default timing it has is the request, which must wait
      // for SThreadBusy
      OCP_TL1_Master_TimingCl my_timing;
      my_timing.RequestGrpStartTime = sthreadbusy_sample_time;
      ocp->setOCPTL1MasterTiming(my_timing);

      // master is timing-sensitive
      ocp->registerTimingSensitiveOCPTL1Master(this);
    }

    // when informed of slave timing, master must re-inform the OCP channel
    // if anything changed
    void setOCPTL1SlaveTiming(OCP_TL1_Slave_TimingCl slave_timing) {
      cout << "  << S-S-T >>   " << name() << endl;
     
      // calculate the sample time for sthreadbusy based on the new timing
      // information from the channel
      sc_time stb_sample = slave_timing.SThreadBusyStartTime + time_quantum;
      // if larger than before, update this module's configuration and
      // then inform the slave via the channel
      if(stb_sample > sthreadbusy_sample_time) {
        sthreadbusy_sample_time = stb_sample;
        OCP_TL1_Master_TimingCl my_timing;
        my_timing.RequestGrpStartTime = sthreadbusy_sample_time;
        ocp->setOCPTL1MasterTiming(my_timing);
      }
    }

  private:
    sc_time sthreadbusy_sample_time;
    sc_time time_quantum;

    int requests_outstanding;
    int *resp_expected;
    int rd_ptr, wr_ptr;
    static const int max_requests_outstanding = 4;

    int requests_sent, responses_got;

    static const double pr_request = 0.125;
    static const double pr_busy = 0.5;
};


// end of multiple inclusion protection
#endif

