///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// (c) Copyright OCP-IP 2003
// OCP-IP Confidential and Proprietary
//
// OCP TL 1 SystemC Channel Model
//
// This version of Channel is derived from the original 
// OCP SLD WG, OCP Transaction Level Layer-1 model by
// Norman Weyrich, Anssi Haverinen, and Joe Chou.
// Please see that model for details.
//
// This version adds full channel access through the ports;
// single function calls to send and receive requests, data, and reponses;
// and clocking to resolve OCP timing.
// These additions are by:
// Alan Kamas, for Sonics Inc., aok@sonicsinc.com, www.kamas.com
// Joe Chou, Sonics Inc., joechou@sonicsinc.com
// along with the support of the OCP SLD Working Group.
//
// $Id: ocp_tl1_channel.h,v 1.1.1.1 2003/12/03 13:57:26 ybajot Exp $
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#ifndef _OCP_TL1_CHANNEL_H
#define _OCP_TL1_CHANNEL_H


#include <math.h>
#include <stdio.h>

// was #include <fstream.h>
#include <fstream>

#include <string>
#include <map>

#include "systemc.h"

#include "tl_channel.h"
#include "ocp_globals.h"
#include "ocp_tl1_ocpmongen_cl.h"
#include "ocp_tl1_master_if.h"
#include "ocp_tl1_slave_if.h"

// Added for gcc 3.3 Compatibility
using namespace std;

// forward declaration needed for the helper class below:
template <class TdataCl> class OCP_TL1_Channel;

// Helper class. Adds a thread process to the channel when 
// OCP monitor generation is needed every clock tick.
// Note: This thread is outside of the channel to work around a problem 
// with SystemC that an SC_THREAD has problems in derived classes.
template<typename TdataCl>
class OCPMonGenClockTick : public sc_module
{
public:
    SC_HAS_PROCESS(OCPMonGenClockTick);
    OCPMonGenClockTick(sc_module_name mn, OCP_TL1_Channel<TdataCl> *myChannel)
        : sc_module(mn), m_channel(myChannel)
    {
        SC_THREAD(AtClockEdge);
    }
    void AtClockEdge(void)
    {
        // nothing to do before the simulation starts.
        m_channel->ocpWait();

        while (true)
        {
            m_channel->OCPClockTick();
            m_channel->ocpWait();
        }
    }
protected:
    OCP_TL1_Channel<TdataCl> *m_channel;
};

// -----------------------------------------------------------------
// -----------------------------------------------------------------

template<typename TdataCl>
class OCP_TL1_Channel :
  public TL_Channel<TdataCl>,
  public OCP_TL1_MasterIF<TdataCl>,
  public OCP_TL1_SlaveIF<TdataCl>
{
public:
  // -----------------------------------------------------------------
  // public types
  // -----------------------------------------------------------------

  typedef typename TdataCl::DataType Td;
  typedef typename TdataCl::AddrType Ta;

  // Needed for when the ocp clock tick process is used
  SC_HAS_PROCESS(OCP_TL1_Channel);

public:
  //---------------------------------------------------------------
  // constructor
  //---------------------------------------------------------------

  OCP_TL1_Channel(std::string name,
                  double clock_period = 1,
                  sc_time_unit clock_time_unit = SC_NS,
                  std::string monFileName = ""
                  )
    : TL_Channel<TdataCl>(name.c_str(),true,true,true),
      m_clkPeriod(clock_period),
      m_clkTimeUnit(clock_time_unit),
      m_ocpMonPtr(NULL),
      m_runCheckPtr(NULL),
      m_addressMask(0x3),
      m_dataMask(0xFFFFFFFF),
      m_clockTickThread(NULL),
      m_exitAfterOCPMon(false)
  {

#ifndef NDEBUG      
    m_doRunCheck = true;
#else
    m_doRunCheck = false;
#endif    

    if ( monFileName.empty() ) {
        // No OCP Monitor file name, so no OCP Monitor
        m_doOCPMon = false;
        m_ocpMonPtr = NULL;
    } else {
        m_doOCPMon = true;
        m_ocpMonPtr = new OCP_TL1_OCPMonGenCl<TdataCl>(name, monFileName); 
    }

    // Clocked Thread to output OCP trace every cycle
    if (m_doOCPMon) {
        // Creates a new, threaded module that will call back to our function
        string threadModuleName = name + "_OCPMonThread";
        m_clockTickThread = new OCPMonGenClockTick<TdataCl>( threadModuleName.c_str(), this );
    }
  }

  OCP_TL1_Channel(std::string name,
                  bool sync,
                  bool use_event = true,
                  bool use_default_event = true,
                  sc_trace_file* vcd_tf = NULL,
                  double clock_period = 1,
                  sc_time_unit clock_time_unit = SC_NS,
                  std::string monFileName = "",
                  bool runtimeCheck = true
                  )
    : TL_Channel<TdataCl>(name.c_str(),sync,use_event,use_default_event),
      m_clkPeriod(clock_period),
      m_clkTimeUnit(clock_time_unit),
      m_ocpMonPtr(NULL),
      m_doRunCheck(runtimeCheck),
      m_runCheckPtr(NULL),
      m_addressMask(0x3),
      m_dataMask(0xFFFFFFFF),
      m_clockTickThread(NULL),
      m_exitAfterOCPMon(false)
  {

    // The OCP Specific TL1 channel only accepts the following:
    assert( sync == true );
    assert( use_event == true );
    assert( use_default_event == true );

    // The OCP TL1 channel no longer accept does trace dumps.
    // This functionality was replaced with the OCP Monitor output.
    // The vcd file name parameter is kept for backward compatibility
    assert( vcd_tf == NULL ); 

    if ( monFileName.empty() ) {
        // No OCP Monitor file name, so no OCP Monitor
        m_doOCPMon = false;
        m_ocpMonPtr = NULL;
    } else {
        m_doOCPMon = true;
        m_ocpMonPtr = new OCP_TL1_OCPMonGenCl<TdataCl>(name, monFileName); 
    }

    // Clocked Thread to output OCP trace every cycle
    if (m_doOCPMon) {
        // Creates a new, threaded module that will call back to our function
        string threadModuleName = name + "_OCPMonThread";
        m_clockTickThread = new OCPMonGenClockTick<TdataCl>( threadModuleName.c_str(), this );
    }
  }

  //---------------------------------------------------------------
  // destructor
  //---------------------------------------------------------------
  virtual ~OCP_TL1_Channel()
  {
      // If this channel had an OCP Monitor, it's time to shut it down.
      if (m_doOCPMon) {
          delete m_ocpMonPtr;
      }
      // If this channel was using an outside, threaded module, time to remove it.
      if (m_clockTickThread) {
          delete m_clockTickThread;
      }
      // If this channel had a runtime checker, it's time to shut it down.
      if (m_doRunCheck) {
          delete m_runCheckPtr;
      }
  }

public:

  // -----------------------------------------------------------------
  // Basic Channel Methods
  // -----------------------------------------------------------------
  virtual void register_port(sc_port_base& port, const char* if_typename)
  { TL_Channel<TdataCl>::register_port(port,if_typename); }

  /////////////////////////////////////////
  // OCP Specific Commands
  /////////////////////////////////////////

  //---------------------------------------------------------------------
  // Configuration Functions
  //---------------------------------------------------------------------

  void 
  setConfiguration( MapStringType& passedMap )
  {
      // Use the passed data base to set up the configuration class
      // NOTE: no "prefix" needed, "name()" passed for debugging
      m_ParamCl->setOCPConfiguration( name(), passedMap );

      // Set up the address mask.
      m_addressMask=0;
      for (int i=0; i<(m_ParamCl->addr_wdth); i++) {
          m_addressMask = m_addressMask << 1;
          m_addressMask = m_addressMask +1;
      }

      // Set up the data mask.
      m_dataMask=0;
      for (int i=0; i<(m_ParamCl->data_wdth); i++) {
          m_dataMask = m_dataMask << 1;
          m_dataMask = m_dataMask +1;
      }

      if (m_doOCPMon)
      {
          // Now that the channel is configured, print out the OCP Monitor header
          m_ocpMonPtr->monitorHeader(m_ParamCl);
      }            

      if (m_doRunCheck)
      {
          // Use the channel configuration to set up the run time checker as well
          if (m_runCheckPtr)
          {
              // TODO: print a warning message here. Why would channel get configured
              //       more than once?
              // There was an old run checker
              delete m_runCheckPtr;
          }
          if (m_doOCPMon) {
              // The run checker should set the passed boolean flag if there is an error.
              // The channel will then print out the last OCP Monitor line before exiting
              // (to help with debugging).
              m_runCheckPtr = new OCP_TL1_CheckerCl<Td,Ta>(m_ParamCl->threads, m_ParamCl->writeresp_enable, name(), &m_exitAfterOCPMon);
          } else {
              // No OCPMonitor in this configuration. If the run checker spots an error it should just exit
              m_runCheckPtr = new OCP_TL1_CheckerCl<Td,Ta>(m_ParamCl->threads, m_ParamCl->writeresp_enable, name(), NULL);
          }
          m_DataCl->setupRunChecker(m_runCheckPtr);
      }
  } 


  //---------------------------------------------------------------------
  // Timing Functions
  //---------------------------------------------------------------------
  //
  void ocpWait(double count = 1) {
      wait(count*m_clkPeriod, m_clkTimeUnit);
  }

  // Handle the case of reset during a longer wait
  // true if successful wait. false if interrupted by reset.
  bool ocpSafeWait(double count = 1) {
      if (m_CommCl->Reset) {
          return false;
      }
      wait(count*m_clkPeriod, m_clkTimeUnit , m_CommCl->ResetStartEvent);
      return (! (m_CommCl->Reset) );
  }

  // This function is called every OCP clock tick
  void 
  OCPClockTick(void)
  {
      // NOTE: would be more efficient to simply change the current MCmd/SResp/MDataValid
      //       and then change it back after the call. 
      //       However, this is cleaner.
      OCPRequestGrp<Td,Ta> monRequest;
      OCPDataHSGrp<Td> monDataHS; 
      OCPResponseGrp<Td> monResponse;

      // if the OCP Monitor is on, dump OCP trace
      if (m_doOCPMon) {

          // Since the default OCP TL1 channel operates without a clock
          // for efficiency, the MCmd, SResp, and MDataVaild signals are
          // dropped to low as soon as the command/response/data are accepted.
          // To be OCP correct, these values should be dropped at the clock edge.
          // If these signals were dropped this cycle, there were saved in
          // special "SavedForOCPTrace" variables so that their values could be
          // restored here, at the clock edge.
          monRequest = m_DataCl->SgetRequestGroup();
          if (m_DataCl->MCmdSavedForOCPTrace != OCP_MCMD_IDLE) {
              monRequest.MCmd = m_DataCl->MCmdSavedForOCPTrace;
              // Value has been used. We can now reset it.
              m_DataCl->MCmdSavedForOCPTrace = OCP_MCMD_IDLE;
          }
          monDataHS = m_DataCl->SgetDataHSGroup();
          if (m_DataCl->MDataValidSavedForOCPTrace != false) {
              monDataHS.MDataValid = m_DataCl->MDataValidSavedForOCPTrace;
              // Value has been used. We can now reset it.
              m_DataCl->MDataValidSavedForOCPTrace = false;
          }
          monResponse = m_DataCl->MgetResponseGroup();
          if (m_DataCl->SRespSavedForOCPTrace != OCP_SRESP_NULL) {
              monResponse.SResp = m_DataCl->SRespSavedForOCPTrace;
              // Value has been used. We can now reset it.
              m_DataCl->SRespSavedForOCPTrace = OCP_SRESP_NULL;
          }

          bool monReqAccept = m_CommCl->RequestEnd;
          bool monRespAccept = m_CommCl->ResponseEnd;
          bool monDataAccept = m_CommCl->DataRequestEnd;

          // NOTE: uses "time -1" as this output pertains to the previous cycle.
          
          m_ocpMonPtr-> monitorOut( m_ParamCl,
                  sc_time_stamp().to_default_time_units() - sc_time(m_clkPeriod,m_clkTimeUnit).to_default_time_units(),
                  monRequest,
                  monReqAccept,
                  monDataHS, 
                  monDataAccept,
                  monResponse,
                  monRespAccept,
                  m_DataCl->MgetLastSThreadBusy(),
                  m_DataCl->MgetLastSDataThreadBusy(),
                  m_DataCl->SgetLastMThreadBusy(),
                  m_DataCl->m_MReset_n,
                  m_DataCl->m_SReset_n,
                  m_DataCl->m_MError,
                  m_DataCl->m_MFlag,
                  m_DataCl->m_SError,
                  m_DataCl->m_SFlag,
                  m_DataCl->m_SInterrupt,
                  m_DataCl->m_Control,
                  m_DataCl->m_ControlWr,
                  m_DataCl->m_ControlBusy,
                  m_DataCl->m_Status,
                  m_DataCl->m_StatusRd,
                  m_DataCl->m_StatusBusy );
      } // end of if (m_doOCPmon)

      // Now that the trace as been dumped, exit if there were any
      // fatal errors during this OCP cycle
      if (m_exitAfterOCPMon) {
          cout << name() << ": Exiting due to earlier errors." << endl;
          exit(1000);
      }
  } 

  //////////////////////////////////////////////////////////////
  // OCP TL1 Methods
  //////////////////////////////////////////////////////////////

  // -----------------------------------------------------------------
  // OCP TL1 commands with overloaded parameters
  // -----------------------------------------------------------------
  bool getOCPResponse( OCPResponseGrp<Td>& myResponse )
  { return getOCPResponse(myResponse, false); }
  bool getOCPResponseBlocking( OCPResponseGrp<Td>& myResponse )
  { return getOCPResponseBlocking(myResponse, false); }
  bool getOCPRequest( OCPRequestGrp<Td,Ta>& myRequest )
  { return getOCPRequest(myRequest, false); }
  bool getOCPRequestBlocking( OCPRequestGrp<Td,Ta>& myRequest )
  { return getOCPRequestBlocking(myRequest, false); }
  bool getOCPDataHS( OCPDataHSGrp<Td>& myData )
  { return getOCPDataHS(myData, false); }
  bool getOCPDataHSBlocking( OCPDataHSGrp<Td>& myData )
  { return getOCPDataHSBlocking(myData, false); }


  //---------------------------------------------------------------------
  // public methods for the OCP Request phase
  //---------------------------------------------------------------------

  bool getSBusy(void) const { return ( (TL_Channel<TdataCl>::MgetSbusy()) || (sc_simulation_time() == m_CommCl->RequestEndTime) ); }

  bool 
  startOCPRequest(OCPRequestGrp<Td,Ta>& newRequest)
  {
      // Is the thread free?
      if (m_ParamCl->sthreadbusy_exact)
      {
          // SThreadBusy is part of the channel and we are required to check it.
          // Now we need to test to see if the thread of the request is busy.
          if ( m_DataCl->isThisThreadBusy(newRequest.MThreadID, getSThreadBusy()) )
          {
              // The thread is busy. Fail.
// #ifndef NDEBUG              
//               cout << name() << ": startOCPRequest failed. It was called for a busy thread on a channel with sthreadbusy_exact." << endl;
// #endif
              return false;
          }
      }
      
      // Has a request already been accepted this cycle? Only one request per ocp cycle is allowed.
      if ( sc_simulation_time() == m_CommCl->RequestEndTime ) {
          return false;
      }

      // Mark that a new request is going on the channel
      if ( !(MputRequest()) )
      {
          return false;
      }

      // Make sure the data & address fits the channel by masking off extra bits.
      newRequest.MData &= m_dataMask;
      newRequest.MAddr &= m_addressMask;

      // Load the new request info onto the channel
      m_DataCl->MputRequestGroup(newRequest);

      return true;
  }
  
  bool startOCPRequestBlocking(OCPRequestGrp<Td,Ta>& newRequest)
  {
      if (m_CommCl->BlockingRequestPending) 
      {
          // Blocking request already started - abort.
          return false;
      }
    
      // Mark that there is a Blocking Request in progress
      m_CommCl->BlockingRequestPending = true;
    
      // Is there a request in progress right now?
      if (m_CommCl->RequestPending || m_CommCl->RequestStart) 
      {
          // There is a request in progress
          // We must wait until it has finished
// #ifndef NDEBUG
//          cout << name() << ": startOCPRequestBlocking is waiting for previous request to end." << endl;
// #endif
          wait(m_CommCl->RequestEndEvent | m_CommCl->ResetStartEvent);

          // Was the channel reset?
          if (m_CommCl->Reset) {
              // In reset. Give up.
              return false;
          }

          // At this point the previous command has been accepted.
          // We now need to wait for the next clock cycle to start a new request.
          ocpWait();
      }

      // The previous request (if any) is over. It's our turn now.
      // Try to send a request
      m_CommCl->BlockingRequestPending = false;
      while ( !startOCPRequest(newRequest) )
      {
          // Was the channel reset?
          if (m_CommCl->Reset) {
              // In reset. Give up.
              return false;
          }
          // The request failed. 
          // Lock up the channel, wait, and try again.
          m_CommCl->BlockingRequestPending = true;
          ocpWait();
          m_CommCl->BlockingRequestPending = false;
      }

      // We are done
      return true;
  }

  bool 
  getSCmdAccept(void) const 
  { 
      return ( !(m_CommCl->RequestPending) && m_CommCl->RequestEnd );
  }

  void
  waitSCmdAccept(void)
  {
    if (!(m_CommCl->RequestPending) && m_CommCl->RequestEnd) {
        // The command has already been accepted
        return;
    } 
    wait(m_CommCl->RequestEndEvent | m_CommCl->ResetStartEvent);
    return;
  }

  void
  waitSDataAccept(void)
  {
    if (!(m_CommCl->DataRequestPending) && m_CommCl->DataRequestEnd) {
        // The data has already been accepted
        return;
    } 
    wait(m_CommCl->DataRequestEndEvent | m_CommCl->ResetStartEvent);
    return;
  }

  void
  waitMRespAccept(void)
  {
    if (!(m_CommCl->ResponsePending) && m_CommCl->ResponseEnd) {
        // The command has already been accepted
        return;
    } 
    wait(m_CommCl->ResponseEndEvent | m_CommCl->ResetStartEvent);
    return;
  }

  unsigned int 
  getSThreadBusy(void) const 
  { 
      // The signal sthreadbusy must be part of the channel
      // for this function to make sense.
      assert( m_ParamCl->sthreadbusy );

      return m_DataCl->MgetSThreadBusy();
  }

  void 
  putSThreadBusy(unsigned int sthreadbusy) 
  { 
      // The signal sthreadbusy must be part of the channel
      // for this function to make sense.
      assert( m_ParamCl->sthreadbusy );

      m_DataCl->SputSThreadBusy( sthreadbusy );
  }

  void 
  putNextSThreadBusy(unsigned int sthreadbusy) 
  { 
      // The signal sthreadbusy must be part of the channel
      // for this function to make sense.
      assert( m_ParamCl->sthreadbusy );

      m_DataCl->SputNextSThreadBusy( sthreadbusy );
  }

  // NOTE: acceptRequest should default to false
  bool getOCPRequest(OCPRequestGrp<Td,Ta>& myRequest, bool acceptRequest) 
  {
      if ( !SgetRequest(acceptRequest) )
      { 
          // Could not get a request. Maybe there is not one available.
          return false;
      }

      // We got the request - now get the request info for the user
      myRequest = m_DataCl->SgetRequestGroup();

      return true;
  }

  // NOTE: acceptRequest should default to false
  bool getOCPRequestBlocking(OCPRequestGrp<Td,Ta>& myRequest, bool acceptRequest) 
  {
      // Is there a blocking get request already in progress? 
      // If so, that getRequestBlocking command has priority to read first.
      if ( m_CommCl->BlockingGetRequestPending ) {
          return false;
      }

      // We are the blocking getRequest. Mark it so that no one else can go.
      m_CommCl->BlockingGetRequestPending = true;

      // Is there a request for us to read?
      if ( !(m_CommCl->RequestStart) || !(m_CommCl->RequestUnread) ) {
          // No request yet. Must be patient.
          wait(m_CommCl->RequestStartEvent | m_CommCl->ResetStartEvent );
          if (m_CommCl->Reset) {
              // The channel has been reset. Give up.
              return false;
          }
      }

      // There should be a request now
      assert( m_CommCl->RequestStart == true );
      assert( m_CommCl->RequestUnread == true );

      // Blocking is over - get the request normally
      m_CommCl->BlockingGetRequestPending = false;
      // get and return the request
      return getOCPRequest(myRequest, acceptRequest);
  }

  bool 
  putSCmdAccept() 
  { 
      // Was this command called at the correct time?
      if (! m_CommCl->RequestStart )
      {
#ifndef NDEBUG
          cout << name() << ": WARNING - putSCmdAccept called when there was no request on the channel." << endl;
#endif
          return false;
      }
      Srelease();
      return true;
  }

  const sc_event& RequestStartEvent(void) const 
  { return (m_CommCl->RequestStartEvent); }

  const sc_event& RequestEndEvent(void) const 
  { return (m_CommCl->RequestEndEvent); }

  //---------------------------------------------------------------------
  // public methods for the OCP Response Phase
  //---------------------------------------------------------------------

  bool getMBusy(void) const { return ( (TL_Channel<TdataCl>::SgetMbusy()) || (sc_simulation_time() == m_CommCl->ResponseEndTime) ); }

  bool startOCPResponse(OCPResponseGrp<Td>& newResponse)
  {
      // Is the thread free?
      if (m_ParamCl->mthreadbusy_exact)
      {
          // MThreadBusy is part of the channel and we are required to check it.
          // Now we need to test to see if the thread is busy.
          if ( m_DataCl->isThisThreadBusy(newResponse.SThreadID, getMThreadBusy()) )
          {
              // The thread is busy. Fail.
#ifdef DEBUG_C
              cout << name() << ": startOCPResponse failed. It was called for a busy thread on a channel with mthreadbusy_exact." << endl;
#endif
              return false;
          }
      }

#ifndef NDEBUG
      // Is the thread free?
      if (m_ParamCl->mthreadbusy)
      {
          // MThreadBusy is part of the channel. 
          // Now we test to see if the thread of the request is busy.
          if ( m_DataCl->isThisThreadBusy(newResponse.SThreadID, getMThreadBusy()) )
          {
              // The thread is busy. Warn the user.
              cout << name() << ": WARNING - startOCPResponse called for a busy thread." << endl;
          }
      }
#endif

      // Has a response already been accepted this cycle? Only one response per ocp cycle is allowed.
      if ( sc_simulation_time() == m_CommCl->ResponseEndTime ) {
          return false;
      }

      // Mark that a new response is going on the channel
      if ( !(SputResponse()) ) {
          // Could not start the response
          return false;
      }

// #ifndef NDEBUG
//    cout << name() << ": startOCPResponse placed the new response on the channel." << endl;
// #endif

      // Make sure the data fits on the channel    
      newResponse.SData &= m_dataMask;

      // The response is ours to send. Put our data unto the channel
      m_DataCl->SputResponseGroup(newResponse);
      
      // All done
      return true;
  }

  bool 
  startOCPResponseBlocking(OCPResponseGrp<Td>& newResponse)
  {
      if (m_CommCl->BlockingResponsePending) {
          // Blocking response already started - abort.
          return false;
      }

      // Mark that there is a Blocking Response in progress
      m_CommCl->BlockingResponsePending = true;

      // Is there a response in progress right now?
      if (m_CommCl->ResponsePending || m_CommCl->ResponseStart) {
          // There is a response in progress
          // We must wait until it has finished
// #ifndef NDEBUG
// cout << name() << ": startOCPResponseBlocking is waiting for previous response to end." << endl;
// #endif
          wait(m_CommCl->ResponseEndEvent | m_CommCl->ResetStartEvent);

          if (m_CommCl->Reset) {
              // Channel was reset while we were waiting. Give up.
              return false;
          }

          // At this point the previous response has been accepted.
          // We now need to wait for the next clock cycle to start a new response
// #ifndef NDEBUG
// cout << name() << ": startOCPResponseBlocking has detected that the previous response has been accepted" << endl;
// #endif
          ocpWait();
      }

      // The previous response (if any) is over. It's our turn now.
      // Try to send a response
      m_CommCl->BlockingResponsePending = false;
      while (!startOCPResponse(newResponse)) {
          // The request failed.

          // Was failure do to a channel reset?
          if (m_CommCl->Reset) {
              // Channel was reset while we were waiting. Give up.
              return false;
          }

          // Lock up the channel, wait and try again.
// #ifndef NDEBUG
// cout << name() << ": startOCPResponseBlocking has failed to put response on the empty channel. Will wait and try again." << endl;
// #endif
          m_CommCl->BlockingResponsePending = true;
          ocpWait();
          m_CommCl->BlockingResponsePending = false;
      }

      // we are done
      return true;
  }

  bool 
  getMRespAccept(void) const 
  { 
      return ( !(m_CommCl->ResponsePending) && m_CommCl->ResponseEnd );
  }


  unsigned int getMThreadBusy(void) const
  { 
      // There is trouble here if this function is being called on a channel
      // that does not have mthreadbusy signal.
      assert( m_ParamCl->mthreadbusy == true);

      return m_DataCl->SgetMThreadBusy();
  }

  void 
  putMThreadBusy(unsigned int mthreadbusy) 
  { 
      // There is trouble here if this function is being called on a channel
      // that does not have mthreadbusy signal.
      assert( m_ParamCl->mthreadbusy == true);

      m_DataCl->MputMThreadBusy( mthreadbusy );
  }

  void 
  putNextMThreadBusy(unsigned int mthreadbusy) 
  { 
      // There is trouble here if this function is being called on a channel
      // that does not have mthreadbusy signal.
      assert( m_ParamCl->mthreadbusy == true);

      m_DataCl->MputNextMThreadBusy( mthreadbusy );
  }


  // NOTE: acceptResponse should default to false
  bool 
  getOCPResponse(OCPResponseGrp<Td>& newResp, bool acceptResponse)
  {
      if ( !MgetResponse(acceptResponse) )  
      {
          // Could not get a response. Maybe there is not one available.
          return false;
      }

      // We got a response: now get the data out of the channel
      newResp = m_DataCl->MgetResponseGroup();

      return true;
  }

  // NOTE: acceptResponse should default to false
  bool 
  getOCPResponseBlocking(OCPResponseGrp<Td>& newResp, bool acceptResponse)
  {
      // Is there a blocking get response already in progress? 
      // If so, that getResponseBlocking command has priority to read first.
      if ( m_CommCl->BlockingGetResponsePending ) {
          return false;
      }

      // We are the blocking getResponse. Mark it so that no one else can go.
      m_CommCl->BlockingGetResponsePending = true;

      // Is there a response for us to read?
      if ( !(m_CommCl->ResponseStart) || !(m_CommCl->ResponseUnread) ) {
          // No response yet. Must be patient.
          wait(m_CommCl->ResponseStartEvent | m_CommCl->ResetStartEvent);
          // Did the channel get reset while we were waiting?
          if (m_CommCl->Reset) {
              // The channel has been reset. Give up.
              return false;
          }
      }

      // There should be a response now
      assert( m_CommCl->ResponseStart == true );
      assert( m_CommCl->ResponseUnread == true );

      // Blocking is over - get the response normally
      m_CommCl->BlockingGetResponsePending = false;
      // get and return the request
      return getOCPResponse(newResp, acceptResponse);
  }

  bool putMRespAccept() 
  { 
      // Was this command called at the correct time?
      if (! m_CommCl->ResponseStart )
      {
#ifndef NDEBUG
          cout << name() << ": WARNING - putMRespAccept called when there was no response on the channel." << endl;
#endif
          return false;
      }
      Mrelease();
      return true;
  }
      
  const sc_event& ResponseStartEvent(void) const 
  { return (m_CommCl->ResponseStartEvent); }

  const sc_event& ResponseEndEvent(void) const 
  { return (m_CommCl->ResponseEndEvent); }

  //---------------------------------------------------------------------
  // public methods for the OCP Data Handshake Phase
  //---------------------------------------------------------------------

  bool getSBusyDataHS(void) const { return ( (TL_Channel<TdataCl>::MgetSbusyData()) || (sc_simulation_time() == m_CommCl->DataRequestEndTime) ); }

  bool
  startOCPDataHS(OCPDataHSGrp<Td>& datahs)
  {
      // Is the thread free?
      if (m_ParamCl->sdatathreadbusy_exact)
      {
          // SDataThreadBusy is part of the channel and we are required to check it.
          // Now we need to test to see if the thread of the data handshake is busy.
          if ( m_DataCl->isThisThreadBusy(datahs.MDataThreadID, getSDataThreadBusy()) )
          {
              // The thread is busy. Fail.
// #ifndef NDEBUG
// cout << name() << ": putData failed. It was called for a busy thread on a channel with sdatathreadbusy_exact." << endl;
// #endif
              return false;
          }
      }

      // Has a data handshake already been accepted this cycle? Only one data handshake per ocp cycle is allowed.
      if ( sc_simulation_time() == m_CommCl->DataRequestEndTime ) {
          return false;
      }

      // Mark that new data is going on the channel
      if ( !(MputDataRequest()) )
      {
          return false;
      }

      // Make sure the data fits on the channel    
      datahs.MData &= m_dataMask;

      // Load the new request info onto the channel
      m_DataCl->MputDataHSGroup(datahs);

      return true;
  }

  bool 
  startOCPDataHSBlocking(OCPDataHSGrp<Td>& newData)
  {
      if (m_CommCl->BlockingDataRequestPending) 
      {
          // Blocking data request already started - abort.
          return false;
      }
    
      // Mark that there is a Blocking Data Request in progress
      m_CommCl->BlockingDataRequestPending = true;
    
      // Is there a data request in progress right now?
      if (m_CommCl->DataRequestPending || m_CommCl->DataRequestStart) 
      {
          // There is a data request in progress
          // We must wait until it has finished
// #ifndef NDEBUG
// cout << name() << ": startOCPDataHSBlocking is waiting for previous data request to end." << endl;
// #endif
          wait(m_CommCl->DataRequestEndEvent | m_CommCl->ResetStartEvent);

          // Did the channel enter reset while we were waiting?
          if (m_CommCl->Reset) {
              // The channel has been reset. Give up.
              return false;
          }

          // At this point the previous data hand shake has been accepted.
          // We now need to wait for the next clock cycle to start new data.
          ocpWait();
      }

      // The previous data hand shake (if any) is over. It's our turn now.
      // Try to send data
      m_CommCl->BlockingDataRequestPending = false;
      while ( !startOCPDataHS(newData) )
      {
          // The data request failed. 

          // Did the channel enter reset while we were waiting?
          if (m_CommCl->Reset) {
              // The channel has been reset. Give up.
              return false;
          }

          // Lock up the channel, wait, and try again.
          // NOTE: failure could be due to thread busy. Are we sure we want to keep trying?
          m_CommCl->BlockingDataRequestPending = true;
          ocpWait();
          m_CommCl->BlockingDataRequestPending = false;
      }

      // We are done
      return true;
  }

  // NOTE: acceptData should default to false
  bool getOCPDataHS(OCPDataHSGrp<Td>& myDataHS, bool acceptData)
  {
      if ( !SgetDataRequest(acceptData) )
      { 
          // Could not get data. Maybe it is not available.
          return false;
      }

      // We got the data - now get the data hand shake info for the user
      myDataHS = m_DataCl->SgetDataHSGroup();

      return true;
  }

  // NOTE: acceptData should default to false
  bool getOCPDataHSBlocking(OCPDataHSGrp<Td>& myDataHS, bool acceptData)
  {
      // Is there a blocking get data already in progress? 
      // If so, that getDataRequestBlocking command has priority to read first.
      if ( m_CommCl->BlockingGetDataRequestPending ) {
          return false;
      }

      // We are the blocking getData. Mark it so that no one else can go.
      m_CommCl->BlockingGetDataRequestPending = true;

      // Is there data for us to read?
      if ( !(m_CommCl->DataRequestStart) || !(m_CommCl->DataRequestUnread) ) {
          // No data yet. Must be patient.
          wait(m_CommCl->DataRequestStartEvent | m_CommCl->ResetStartEvent);
          // Did the channel enter reset while we were waiting?
          if (m_CommCl->Reset) {
              // The channel has been reset. Give up.
              return false;
          }
      }

      // There should be data now
      assert( m_CommCl->DataRequestStart == true );
      assert( m_CommCl->DataRequestUnread == true );

      // Blocking is over - get the data normally
      m_CommCl->BlockingGetDataRequestPending = false;
      // get and return the data
      return getOCPDataHS(myDataHS, acceptData);
  }

  bool putSDataAccept() 
  { 
      // Was this command called at the correct time?
      if (! m_CommCl->DataRequestStart )
      {
#ifndef NDEBUG
          cout << name() << ": WARNING - putSDataAccept called when there was no data on the channel." << endl;
#endif
          return false;
      }
      SreleaseData();
      return true;
  }

  bool getSDataAccept(void) const 
  { 
      return ( !(m_CommCl->DataRequestPending) && m_CommCl->DataRequestEnd );
  }

  // Optional command provided to the verify the timing of the Master
  // NOTE: doesn't actually do anything
  // NOTE: Not required.
  void putNullMDataValid(void) const
  { 
#ifndef NDEBUG
      if (m_DataCl->m_DataHSGrp[1 - m_DataCl->m_DataHSToggle].MDataValid == false )
      {
          cout << name() << ": MDataValid is set to Null" << endl;
      } else {
          cout << name() << ": WARNING - putNullMDataValid called and MDataValid is not false (Data Request still in progress?)" << endl;
      }
#endif
  }

  const sc_event& DataHSStartEvent(void) const 
  { return (m_CommCl->DataRequestStartEvent); }

  const sc_event& DataHSEndEvent(void) const 
  { return (m_CommCl->DataRequestEndEvent); }

  void putSDataThreadBusy(unsigned int sdatathreadbusy) 
  { 
      // The sdatathreadbusy signal needs to be part of the 
      // channel for this function to work
      assert(m_ParamCl->sdatathreadbusy == true);

      m_DataCl->SputSDataThreadBusy( sdatathreadbusy );
  }

  void 
  putNextSDataThreadBusy(unsigned int newValue) 
  { 
      // The sdatathreadbusy signal needs to be part of the 
      // channel for this function to work
      assert(m_ParamCl->sdatathreadbusy == true);

      m_DataCl->SputNextSDataThreadBusy( newValue );
  }


  unsigned int getSDataThreadBusy(void) const
  { 
      // The sdatathreadbusy signal must be part of the channel for this command
      // to function correctly
      assert( m_ParamCl->sdatathreadbusy == true );

      return m_DataCl->MgetSDataThreadBusy();
  }

  //---------------------------------------------------------------------
  // public Reset Commands
  //---------------------------------------------------------------------

  // resets the channel (only to be called internally)
  void reset(void)
  { TL_Channel<TdataCl>::reset(); }

  bool getReset(void)
  { return (m_CommCl->Reset); }

  const sc_event& ResetStartEvent(void) const 
  { return (m_CommCl->ResetStartEvent); }

  const sc_event& ResetEndEvent(void) const 
  { return (m_CommCl->ResetEndEvent); }

  void MResetAssert(void)
  { 
      if (! m_DataCl->m_MReset_n) {
          // Master has already caused a reset.
          // Do nothing.
          return;
      }

      assert(m_ParamCl->mreset);

      m_DataCl->m_MReset_n = false;
      m_DataCl->m_MResetStartTime= sc_time_stamp();

      // TODO: This should be a separate function
      // If the channel is not already in reset, reset it.
      if (! m_CommCl->Reset) {
          // reset the channel
          m_CommCl->Reset = true;
          reset();
          m_runCheckPtr->startReset();
          m_CommCl->ResetStartEvent.notify();
      }
  }
      
  void MResetDeassert(void) 
  {
      if (m_DataCl->m_MReset_n) {
          // Master is already out of reset
          // Do nothing.
          return;
      }

      assert(m_ParamCl->mreset);

      m_DataCl->m_MReset_n = true;

      // This command can only be called 16 or more cycles after reset was asserted
      assert( (sc_time_stamp() - m_DataCl->m_MResetStartTime) > sc_time(15.99 * m_clkPeriod, m_clkTimeUnit) );

      if (m_DataCl->m_SReset_n) {
          // The slave reset has also been deasserted. Reset is over.
          assert(m_CommCl->Reset);  // Channel should be in reset mode.

          // reset the channel again to make sure all signals are re-initialized
          reset();
          m_runCheckPtr->endReset();
          m_CommCl->ResetStartEvent.notify();
          // Now pull out of reset mode
          m_CommCl->Reset = false;
          m_CommCl->ResetEndEvent.notify();
      }
  }

  void SResetAssert(void)
  { 
      if (! m_DataCl->m_SReset_n) {
          // Slave has already caused a reset.
          // Do nothing.
          return;
      }

      assert(m_ParamCl->sreset);

      m_DataCl->m_SReset_n = false;
      m_DataCl->m_SResetStartTime= sc_time_stamp();

      // TODO: This should be a separate function
      // If the channel is not already in reset, reset it.
      if (! m_CommCl->Reset) {
          // reset the channel
          m_CommCl->Reset = true;
          m_runCheckPtr->startReset();
          reset();
          m_CommCl->ResetStartEvent.notify();
      }
  }
      
  void SResetDeassert(void)
  {
      if (m_DataCl->m_SReset_n) {
          // Slave is already out of reset
          // Do nothing.
          return;
      }

      assert(m_ParamCl->sreset);

      m_DataCl->m_SReset_n = true;

      // This command can only be called 16 or more cycles after reset was asserted
      assert( (sc_time_stamp() - m_DataCl->m_SResetStartTime) > sc_time(15.99 * m_clkPeriod, m_clkTimeUnit) );

      if (m_DataCl->m_MReset_n) {
          // The master reset has also been deasserted. Reset is over.
          assert(m_CommCl->Reset);  // Channel should be in reset mode.

          // reset the channel again to make sure all signals are re-initialized
          reset();
          m_runCheckPtr->endReset();
          // Now pull out of reset mode
          m_CommCl->Reset = false;
          m_CommCl->ResetEndEvent.notify();
      }
  }

  //---------------------------------------------------------------------
  // public OCP Master Sideband Commands
  //---------------------------------------------------------------------
  void MputMError(bool nextValue) 
  { m_DataCl->MputMError(nextValue); }
  void MputMFlag(unsigned int nextValue)
  { m_DataCl->MputMFlag(nextValue); }
  bool MgetSError(void) const
  { return (m_DataCl->MgetSError()); }
  unsigned int MgetSFlag(void) const
  { return (m_DataCl->MgetSFlag()); }
  bool MgetSInterrupt(void) const
  { return (m_DataCl->MgetSInterrupt()); }

  //---------------------------------------------------------------------
  // public OCP Master Sideband Events 
  //---------------------------------------------------------------------
  const sc_event& SidebandSErrorEvent(void) const  
  { return (m_DataCl->SidebandSErrorEvent()); }
  const sc_event& SidebandSFlagEvent(void) const
  { return (m_DataCl->SidebandSFlagEvent()); }
  const sc_event& SidebandSInterruptEvent(void) const
  { return (m_DataCl->SidebandSInterruptEvent()); }

  //---------------------------------------------------------------------
  // public OCP Slave Sideband Commands
  //---------------------------------------------------------------------
  bool SgetMError(void) const
  { return (m_DataCl->SgetMError()); }
  unsigned int SgetMFlag(void) const
  { return (m_DataCl->SgetMFlag()); }
  void SputSError(bool nextValue)
  { m_DataCl->SputSError(nextValue); }
  void SputSFlag(unsigned int nextValue)
  { m_DataCl->SputSFlag(nextValue); }
  void SputSInterrupt(bool nextValue)
  { m_DataCl->SputSInterrupt(nextValue); }

  //---------------------------------------------------------------------
  // public OCP Slave Sideband Events
  //---------------------------------------------------------------------
  const sc_event& SidebandMErrorEvent(void) const
  { return (m_DataCl->SidebandMErrorEvent()); }
  const sc_event& SidebandMFlagEvent(void) const
  { return (m_DataCl->SidebandMFlagEvent()); }

  //---------------------------------------------------------------------
  // public OCP System Sideband Commands
  //---------------------------------------------------------------------
  bool SysputControl(int nextValue)
  { return m_DataCl->SysputControl(nextValue); }
  bool SysgetControlBusy(void) const
  { return (m_DataCl->SysgetControlBusy()); }
  void SysputControlWr(bool nextValue)
  { m_DataCl->SysputControlWr(nextValue); }
  int SysgetStatus(void) const
  { return (m_DataCl->SysgetStatus()); }
  bool readStatus(int& currentValue) const
  { return (m_DataCl->readStatus(currentValue)); }
  bool SysgetStatusBusy(void) const
  { return (m_DataCl->SysgetStatusBusy()); }
  void SysputStatusRd(bool nextValue)
  { m_DataCl->SysputStatusRd(nextValue); }

  //---------------------------------------------------------------------
  // public OCP System Sideband Events
  //---------------------------------------------------------------------
  const sc_event& SidebandControlBusyEvent(void) const
  { return (m_DataCl->SidebandControlBusyEvent()); }
  const sc_event& SidebandStatusEvent(void) const
  { return (m_DataCl->SidebandStatusEvent()); }
  const sc_event& SidebandStatusBusyEvent(void) const
  { return (m_DataCl->SidebandStatusBusyEvent()); }

  //---------------------------------------------------------------------
  // public OCP Core Sideband Commands
  //---------------------------------------------------------------------
  int CgetControl(void) const
  { return (m_DataCl->CgetControl()); }
  void CputControlBusy(bool nextValue)
  { m_DataCl->CputControlBusy(nextValue); }
  bool CgetControlWr(void) const
  { return (m_DataCl->CgetControlWr()); }
  void CputStatus(int nextValue)
  { m_DataCl->CputStatus(nextValue); }
  void CputStatusBusy(bool nextValue)
  { m_DataCl->CputStatusBusy(nextValue); }
  bool CgetStatusRd(void) const
  { return (m_DataCl->CgetStatusRd()); }
                                      
  //---------------------------------------------------------------------
  // public OCP Core Sideband Events
  //---------------------------------------------------------------------
  const sc_event& SidebandControlEvent(void) const 
  { return (m_DataCl->SidebandControlEvent()); }
  const sc_event& SidebandControlWrEvent(void) const
  { return (m_DataCl->SidebandControlWrEvent()); }
  const sc_event& SidebandStatusRdEvent(void) const
  { return (m_DataCl->SidebandStatusRdEvent()); }


  ///////////////////////////////////
  // Port Access to the Generic commands in the underlying Generic Channel
  // Generic commands kept for backward compatibility
  ///////////////////////////////////

  // -----------------------------------------------------------------
  // Generic Direct Channel Access Methods
  // -----------------------------------------------------------------
  virtual bool MputDirect(int MasterID, bool IsWrite, Td *Data, Ta Address,
                          int NumWords)
  {
    return (TL_Channel<TdataCl>::MputDirect(MasterID,IsWrite,Data,Address,NumWords));
  }
  virtual bool SputDirect(int SlaveID, bool IsWrite, Td *Data, Ta Address,int NumWords)
  {
    return (TL_Channel<TdataCl>::SputDirect(SlaveID,IsWrite,Data,Address,NumWords));
  }

  // -----------------------------------------------------------------
  // Direct Access to the internals of the channel.
  // Necessary for the use of the Generic Commands.
  // Note: use with caution.
  // -----------------------------------------------------------------
  TdataCl* GetDataCl(void) 
  { return (TL_Channel<TdataCl>::GetDataCl()); }
  ParamCl<TdataCl>* GetParamCl(void)
  { return (TL_Channel<TdataCl>::GetParamCl()); }
  CommCl* GetCommCl(void) 
  { return (TL_Channel<TdataCl>::GetCommCl()); }

  // -----------------------------------------------------------------
  // Generic Channel Methods
  // -----------------------------------------------------------------
  void Mrelease(sc_time time) { TL_Channel<TdataCl>::Mrelease(time); }
  void Srelease(sc_time time) { TL_Channel<TdataCl>::Srelease(time); }
  void Mrelease(void) { TL_Channel<TdataCl>::Mrelease(); }
  void Srelease(void) { TL_Channel<TdataCl>::Srelease(); }
  void SreleaseData(sc_time time) { TL_Channel<TdataCl>::SreleaseData(time); }
  void SreleaseData(void) { TL_Channel<TdataCl>::SreleaseData(); }
  void MreleasePE(void) { TL_Channel<TdataCl>::MreleasePE(); }
  void SreleasePE(void) { TL_Channel<TdataCl>::SreleasePE(); }
  void SreleaseDataPE(void) { TL_Channel<TdataCl>::SreleaseDataPE(); }
  bool MgetSbusy(void) const { return (TL_Channel<TdataCl>::MgetSbusy()); }
  bool MgetSbusyData(void) const { return (TL_Channel<TdataCl>::MgetSbusyData()); }
  bool SgetMbusy(void) const { return (TL_Channel<TdataCl>::SgetMbusy()); }
  bool IsWrite(void) { return (TL_Channel<TdataCl>::IsWrite()); }
  void SregisterDirectIF(SdirectIF<TdataCl>* A)
  { TL_Channel<TdataCl>::SregisterDirectIF(A); }
  void MregisterDirectIF(MdirectIF<TdataCl>* A)
  { TL_Channel<TdataCl>::MregisterDirectIF(A); }
  bool MputWriteRequest(void)
  { return (TL_Channel<TdataCl>::MputWriteRequest()); }
  bool MputReadRequest(void)
  { return (TL_Channel<TdataCl>::MputReadRequest()); }
  bool MputWriteRequestBlocking(void) 
  { return (TL_Channel<TdataCl>::MputWriteRequestBlocking()); }
  bool MputReadRequestBlocking(void) 
  { return (TL_Channel<TdataCl>::MputReadRequestBlocking()); }
  bool MputRequest(void) { return (TL_Channel<TdataCl>::MputRequest()); }
  bool MputRequestBlocking(void)
  { return (TL_Channel<TdataCl>::MputRequestBlocking()); }
  bool SgetRequest(bool Release)
  { return (TL_Channel<TdataCl>::SgetRequest(Release)); }
  bool SgetRequestBlocking(bool Release)
  { return (TL_Channel<TdataCl>::SgetRequestBlocking(Release)); }
  bool SgetRequestPE(void) { return (TL_Channel<TdataCl>::SgetRequestPE()); }
  bool SputResponse(void) { return (TL_Channel<TdataCl>::SputResponse()); }
  bool SputResponseBlocking(void)
  { return (TL_Channel<TdataCl>::SputResponseBlocking()); }
  bool MgetResponse(bool Release)
  { return (TL_Channel<TdataCl>::MgetResponse(Release)); }
  bool MgetResponseBlocking(bool Release)
  { return (TL_Channel<TdataCl>::MgetResponseBlocking(Release)); }
  bool MgetResponsePE(void)
  { return (TL_Channel<TdataCl>::MgetResponsePE()); }
  bool MputDataRequest(void)
  { return (TL_Channel<TdataCl>::MputDataRequest()); }
  bool MputDataRequestBlocking(void)
  { return (TL_Channel<TdataCl>::MputDataRequestBlocking()); }
  bool SgetDataRequest(bool Release)
  { return (TL_Channel<TdataCl>::SgetDataRequest(Release)); }
  bool SgetDataRequestBlocking(bool Release)
  { return (TL_Channel<TdataCl>::SgetDataRequestBlocking(Release)); }
  bool SgetDataRequestPE(void)
  { return (TL_Channel<TdataCl>::SgetDataRequestPE()); }

  // -----------------------------------------------------------------
  // Generic Channel Events
  // -----------------------------------------------------------------
  const sc_event& default_event() const
  { return (TL_Channel<TdataCl>::default_event()); }


protected:
  //---------------------------------------------------------------
  // protected methods
  //---------------------------------------------------------------

  // display a fatal error message and quick the program
  void fatalError(char* sub_title, char* msg)
  {
    cout << endl << name() << ": ERROR: " << sub_title << ":" << msg << endl;
    assert(0);
  }

protected:
  //----------------------------------------------------------------------
  // member data
  //----------------------------------------------------------------------

  // OCP clock information
  double       m_clkPeriod;
  sc_time_unit m_clkTimeUnit;

  // OCP Trace variables
  bool m_doOCPMon;
  OCP_TL1_OCPMonGenCl<TdataCl> *m_ocpMonPtr;

  // RunTime Checker Flag
  bool m_doRunCheck;    // Do runtime checking if true
  OCP_TL1_CheckerCl<Td,Ta> *m_runCheckPtr;

  // Address Mask - used to make sure user address is not to big for the channel.
  Ta m_addressMask;  

  // Data Mask - used to make sure user data is not to big for the channel.
  Td m_dataMask;  

  // Threaded Module for timing the output of OCP Monitor data
  OCPMonGenClockTick<TdataCl> *m_clockTickThread;

  // Flag - set if a fatal error has occured and the simulation should be stopped
  // once the last line of the OCP Trace File has been output (for debugging).
  bool m_exitAfterOCPMon;

};

#endif  // _OCP_TL1_CHANNEL_H
