// 
//  Copyright 2003 OCP-IP
//  OCP-IP Confidential & Proprietary
//
// ============================================================================
//      Project : OCP SLD WG, OCP Transaction Level
//       Author : Norman Weyrich, Synopsys Inc., weyrich@synopsys.com
//                (modified by Joe Chou, Sonics Inc., joechou@sonicsinc.com)
//                (modified by Yann Bajot, Prosilog, bajot@prosilog.com)
//         Date : 11/13/2003
//
//  Description : Transaction Level - Layer-2 example Slave
//                Features: 
//                - Blocking methods are used.
//                - Slave puts data into his memory (write request)
//                  or sends data from his memory to the Master (read request).
//                - Slave can be pipelined or non-pipelined.
//                - In the pielined case the Slave uses FIFO to not drop
//                  requests.
//                
// Parameter    : 
//   Template arguments.
//   - TdataCl: Data class containing data members and
//              access methods for the data exchanged between
//              Masters and Slaves
//   - Td: Data type
//   - Ta: Address type
//                
//   Constructor arguments.
//     - sc_module_name: Name of the module instance
//     - ID: Unique number identifying the Master.
//           ID must be unique among all Masters attached to the same Bus.
//     - StartAddress: First address of the Slaves memory.
//     - EndAddress: Last address of the Slaves memory.
//                   The address range of all Slaves in a system
//                   must be mutually distinct.
//     - Pipelined: Switch to change between non-pipelined and
//                  pipelined operation mode of the Master.
//     - WriteRequest: Switch to enable/disable the sending of a
//                     response packet to a Master write request.
//                     Note that all modules involved in a system
//                     must use the same value.
//     - ReadAcceptCycles: Number of cycles per beat the Slave waits
//                         before accepting a read request.
//     - WriteAcceptCycles: Number of cycles per beat the Slave waits
//                          before accepting a write request.
//     - ReadResponseCycles: Number of cycles per beat the Slave waits
//                           before starting a read response.
//     - WriteResponseCycles: Number of cycles per beat the Slave waits
//                            before starting a write response.
//
// ============================================================================

// Pipelined versus non-pipelined operation mode
// ---------------------------------------------
// The Slave can be pipelined or non-pipelined.
// In the non-pipelined case the requests and responses
// are handled sequentially in a single thread. This is pretty secure
// against deadlocks.
// In the piplined case the request and reponse methods are called from
// two different threads. The response thread starts after the request
// thread has triggered an event.
//
// Timing
// ------
// The Slave has the following timing:
// Non-pipelined case:
// - Write request transfer: 
//         Accept  : wait NumWords * WriteAcceptCycles cycles
//         Response: wait WriteResponseCycles cycles
// - Read  request  transfer:
//         Accept  : wait ReadAcceptCycles cycles
//         Response: wait NumWords * ReadResponseCycles cycles
// Pipelined case:
// - Write request transfer: 
//         Accept  : wait NumWords * WriteAcceptCycles cycles
//         Response: wait WriteResponseCycles cycles
// - Read  request  transfer:
//         Accept  : wait ReadAcceptCycles cycles
//         Response: wait NumWords * ReadResponseCycles cycles
// Note that in the non-pipelined case the response is started after
// the request, while in the pipelined case request and response are started
// in parallel.
// ============================================================================

#include "ocp_tl2_slave.h"

// ----------------------------------------------------------------------------
// Process : OCP_TL2_Slave::OCP_TL2_Slave
// 
// Constructor
// ----------------------------------------------------------------------------
template<class TdataCl>
OCP_TL2_Slave<TdataCl>::OCP_TL2_Slave(sc_module_name name_, int ID,
  Ta StartAddress, Ta EndAddress, bool Pipelined, bool WriteResponse,
  int ReadAcceptCycles, int WriteAcceptCycles, int ReadResponseCycles,
  int WriteResponseCycles)
   : sc_module(name_),
     SlaveP("SlavePort"),
     m_ID(ID),
     m_StartAddress(StartAddress),
     m_EndAddress(EndAddress),
     m_Pipelined(Pipelined),
     m_WriteResponse(WriteResponse),
     m_ReadAcceptCycles(ReadAcceptCycles),
     m_WriteAcceptCycles(WriteAcceptCycles),
     m_ReadResponseCycles(ReadResponseCycles),
     m_WriteResponseCycles(WriteResponseCycles),
     m_ReqTaNum(0),
     m_ResTaNum(0)
{
  int i;

  // InitParameters();

  //
  m_NumBytesPerWord = sizeof(Td);
  m_SmemoryLength = (m_EndAddress - m_StartAddress + 1) / m_NumBytesPerWord;

  // allcoate memory
  if (m_SmemoryLength > 0) {
    try {
      m_SlaveMemory = new Td[m_SmemoryLength];
    }
    catch(std::bad_alloc ex) {
      cerr << "ERROR: Memory allocation for Slave memory failed "
            << name() << ">" << endl;
      cerr << "       Size  is: " << m_SmemoryLength << endl;
      cerr << "       Error is: " << ex.what() << endl;
      sc_stop();
      return;
    }
  }

  for (i = 0; i < m_SmemoryLength; i++)
    m_SlaveMemory[i] = 0;

  for (i = 0; i < TL_SLAVE_FIFO_DEPTH; i++)
    m_SlaveRespWait[i] = true;

  //
  if (m_Pipelined) {
    SC_THREAD(SlaveRequest);
    SC_THREAD(SlaveResponse);
    sensitive << m_SlaveRespEvent;
  } else {
    SC_THREAD(Slave);
  }
}

// ----------------------------------------------------------------------------
// Process : OCP_TL2_Slave::~OCP_TL2_Slave
// 
// Destructor
// ----------------------------------------------------------------------------
template<class TdataCl>
OCP_TL2_Slave<TdataCl>::~OCP_TL2_Slave()
{
  delete [] m_SlaveMemory;
}

// ----------------------------------------------------------------------------
//  Method : OCP_TL2_Slave::Slave()
//
//  Top level method of the Slave (non-pipelined case)
//
// ----------------------------------------------------------------------------
template<class TdataCl>
void OCP_TL2_Slave<TdataCl>::Slave()
{
  Ta            Address;
  unsigned int           NumWords;
  bool chunk_last;
  bool          Write;
  OCPSRespType  Sresp;
  int           Sconnid;

  // assuming the master side will pass in valid data pointers
  Td*           Mdata;
  Td*           Sdata;

  while (true) {
    // Get a new OCP request; use blocking methods
    if ((SlaveP->SgetRequestBlocking(false))) {
      // no release

      // retrieve the request
      Write = SlaveP->IsWrite();
      Address = m_DataCl->SgetMAddr();

      //
      if (Write) {
        // write request
 
        // retrieve the write data and OCP word length
        Mdata = m_DataCl->SgetMData(NumWords,chunk_last);
        Sconnid = m_DataCl->SgetMConnID();

        // Acceptance delay
        wait(NumWords*m_WriteAcceptCycles,TL_TIME_SCALE);

#ifdef DEBUG_S
        PrintTrans(m_ResTaNum,Address,NumWords," Blocking Response ");
#endif
#ifdef DEBUG_G1
        PrintTime("Accept write request in Slave ",m_ID,Address);
#endif

        // accept the write request

        // write data into the memory (the direct method call is just
        // a reuse of an existing method, and there is nothing to do with
        // the debug interface).
        Sresp = (MputDirect(m_BusID,Write,Mdata,Address,NumWords))?
                OCP_SRESP_DVA : OCP_SRESP_ERR;

        // Release the OCP request channel
        SlaveP->Srelease();

        // generate a write response, if needed
        if (m_WriteResponse) {
          // Response delay
          wait(m_WriteResponseCycles,TL_TIME_SCALE);

          // response phase

          // setup a write response
          m_DataCl->SputSResp(Sresp);

          // uses the srespinfo field to transmit the 'response' connID
          m_DataCl->SputSRespInfo(Sconnid);

          // Inform master that this is a Write response
          // To do so, we use the sdatainfo field
          m_DataCl->SputSDataInfo(1);

#ifdef DEBUG_G1
          PrintTime("Start write response in Slave ",m_ID,Address);
#endif
 
          // Send write response
          if (!SlaveP->SputResponseBlocking()) {
            // Deep ship: Should never happen
            cout << "Error: Master (" << m_BusID << ") dropped Slave ("
                 << m_ID << ") response <" << name() << ">" << endl;
          }
          m_ResTaNum++;
        }
      } else {
        // read request
 
        // retrieve the read request OCP word length and a return data pointer
        Sdata = m_DataCl->SgetMData(NumWords);
        Sconnid = m_DataCl->SgetMConnID();

        // Acceptance delay
        wait(m_ReadAcceptCycles,TL_TIME_SCALE);

#ifdef DEBUG_G1
        PrintTime("Accept read request in Slave ",m_ID,Address);
#endif
#ifdef DEBUG_S
        PrintTrans(m_ResTaNum,Address,NumWords," Blocking Response ");
#endif

        // accept the read request

        // read the Data (the direct method call is just a reuse of an
        // existing method, and there is nothing to do with the debug
        // interface).
        Sresp = (MputDirect(m_BusID,Write,Sdata,Address,NumWords))?
                OCP_SRESP_DVA : OCP_SRESP_ERR;

        // Release the OCP request Channel
        SlaveP->Srelease();

        // Response delay
        wait(NumWords*m_ReadResponseCycles,TL_TIME_SCALE);

        // Response phase

#ifdef DEBUG_G1
        PrintTime("Start read response in Slave ",m_ID,Address);
#endif

        // setup a read response
        m_DataCl->SputSResp(Sresp);
        m_DataCl->SputSData(Sdata,NumWords);

        // uses the srespinfo field to transmit the 'response' connID
        m_DataCl->SputSRespInfo(Sconnid);

        // Inform master that this is a Read response
        // To do so, we use the sdatainfo field
          m_DataCl->SputSDataInfo(0);

        // Send response to Master/Bus
        if (!SlaveP->SputResponseBlocking()) {
          // Deep ship: Should never happen
          cout << "Error: Master (" << m_BusID << ") dropped Slave ("
               << m_ID << ") response <" << name() << ">" << endl;
        }

        m_ResTaNum++;
      } // end of read branch
    } // end of SgetRequest() bransch
  } // end of while (true) loop
} // end of Slave method

// ----------------------------------------------------------------------------
//  Method : OCP_TL2_Slave::SlaveRequest()
//
//  Top level request method of the Slave (pipelined case)
//
// ----------------------------------------------------------------------------
template<class TdataCl>
void OCP_TL2_Slave<TdataCl>::SlaveRequest()
{
  int NumWords;
  int ReqIdx = 0;
  bool chunk_last;

  while (true) {
    // make sure there is a space left for another outstanding request
    while (m_ReqTaNum - m_ResTaNum >= TL_SLAVE_FIFO_DEPTH) {
      // FIFO full - need to work on the response backlog

#ifdef DEBUG_S
      cout << "Slave (" << m_ID << ") waits to process responses <"
           << name() << ">, m_ReqTaNum: " << m_ReqTaNum
           << " m_ResTaNum:" << m_ResTaNum << endl;
#endif

      // skip a few cycles
      wait(TL_SLAVE_FIFO_DEPTH*(1+m_ReadResponseCycles),TL_TIME_SCALE);
    }

    // use the blocking method to get a new request
    if ((SlaveP->SgetRequestBlocking(false))) {
      // no release

      // get the new request and store it in the FIFO
      m_Write[ReqIdx] = SlaveP->IsWrite();
      m_Address[ReqIdx] = m_DataCl->SgetMAddr();
      m_MconnID[ReqIdx] = m_DataCl->SgetMConnID();

      // the master passes along either a write data pointer or a
      // to-be-returned read data pointer
      m_Mdata[ReqIdx] = m_DataCl->SgetMData(m_NumWords[ReqIdx],chunk_last);


#ifdef DEBUG_S
      PrintTrans(m_ReqTaNum,m_Address[ReqIdx],m_NumWords[ReqIdx],
                 " Blocking Request ");
#endif

      // Put data into the memory (write) or into the Data field (read).
      // The direct method call is just a reuse of an existing method.
      // This has nothing to do with the debug interface.
      m_Sresp[ReqIdx] = (MputDirect(m_BusID,m_Write[ReqIdx],
                        m_Mdata[ReqIdx],m_Address[ReqIdx],m_NumWords[ReqIdx]))?
                        OCP_SRESP_DVA: OCP_SRESP_ERR;

      //
      if (m_Write[ReqIdx]) {
        // write request

        // Acceptance delay
        int AcceptDelay = NumWords * m_WriteAcceptCycles;

        // schedule a release of the OCP request channel (i.e., assert the
        // SCmdAccept signal) at a delta time later
        SlaveP->Srelease(sc_time(AcceptDelay,TL_TIME_SCALE));

#ifdef DEBUG_G1
        PrintFuTime("Accept write request in Slave ",m_ID,m_Address[ReqIdx],
                    (double)AcceptDelay*TL_TIME_SCALE_SEC);
#endif

        // check whether need to generate a write response
        if (m_WriteResponse) {
          // setup a response and notify the response thread

          // compute pipelined response delay and trigger response thread
          sc_time ResponseDelay = sc_time(m_WriteResponseCycles,
                                          TL_TIME_SCALE);
          m_RespStartTime[ReqIdx] = sc_time_stamp() + ResponseDelay;

          // schedule a future notification to the response thread
          m_SlaveRespWait[ReqIdx] = false;
          m_SlaveRespEvent.notify(ResponseDelay);

#ifdef DEBUG_S1
          cout << "Slave (" << m_ID << ") starts response process at "
               << m_RespStartTime[ReqIdx].to_double() << "  <"
               << name() << ">" << endl;
#endif

          // advance the outstanding request's FIFO index
          ReqIdx++;
          if (ReqIdx >= TL_SLAVE_FIFO_DEPTH)
            ReqIdx = 0;

          // advance the outstanding request counter
          m_ReqTaNum++;
        }
      } else {
        // read request

        // Acceptance delay
        int AcceptDelay = m_ReadAcceptCycles;

        // schedule a release of the OCP request channel (i.e., assert the
        // SCmdAccept signal) at a delta time later
        SlaveP->Srelease(sc_time(AcceptDelay,TL_TIME_SCALE));

#ifdef DEBUG_G1
        PrintFuTime("Accept read request in Slave ",m_ID,m_Address[ReqIdx],
                    (double)AcceptDelay*TL_TIME_SCALE_SEC);
#endif
        // setup a response and notify the response thread

        // compute pipelined response delay and trigger response thread
        sc_time ResponseDelay = sc_time(NumWords*m_ReadResponseCycles,
                                        TL_TIME_SCALE);
        m_RespStartTime[ReqIdx] = sc_time_stamp() + ResponseDelay;

        // schedule a future notification to the response thread
        m_SlaveRespWait[ReqIdx] = false;
        m_SlaveRespEvent.notify(ResponseDelay);

#ifdef DEBUG_S1
        cout << "Slave (" << m_ID << ") starts response process at "
             << m_RespStartTime[ReqIdx].to_double() << "  <"
             << name() << ">" << endl;
#endif

        // advance the outstanding request's FIFO index
        ReqIdx++;
        if (ReqIdx >= TL_SLAVE_FIFO_DEPTH)
          ReqIdx = 0;

        // advance the outstanding request counter
        m_ReqTaNum++;
      }
    } // end of Sget  = true branch
  } // end of while (true) loop
} // end of method

// ----------------------------------------------------------------------------
//  Method : OCP_TL2_Slave::SlaveResponse()
//
//  Top level response method of the Slave (pipelined case)
//
// ----------------------------------------------------------------------------
template<class TdataCl>
void OCP_TL2_Slave<TdataCl>::SlaveResponse()
{
  int ResIdx = 0;

  while (true) {
    // use blocking methods

#ifdef DEBUG_S1
    cout << "Slave (" << m_ID << ", " << ResIdx << ") starts waiting at "
         << sc_time_stamp().to_double() << "  <"
           << name() << ">" << endl;
#endif

    // wait for a notification event
    if (m_SlaveRespWait[ResIdx]) wait();

    // check if we still have to wait
    sc_time CurrentTime = sc_time_stamp();

#ifdef DEBUG_S1
    sc_time ti1 = (m_RespStartTime[ResIdx] > CurrentTime)?
                  m_RespStartTime[ResIdx] - CurrentTime : SC_ZERO_TIME;
    cout << "Slave (" << m_ID << ", " << ResIdx << ") waits "
         << ti1.to_double() << " in response process at "
         << CurrentTime.to_double() << "  <" << name() << ">" << endl;
#endif

    if (m_RespStartTime[ResIdx] > CurrentTime)
      wait(m_RespStartTime[ResIdx] - CurrentTime);

#ifdef DEBUG_S
    PrintTrans(m_ResTaNum,m_Address[ResIdx],m_NumWords[ResIdx],
               " Blocking Response ");
#endif

    // ----------------
    // setup a response
    // ----------------

    // set the response and address filed
    m_DataCl->SputSResp(m_Sresp[ResIdx]);

    // transmit the response connid via the SRespInfo field
    m_DataCl->SputSRespInfo(m_MconnID[ResIdx]);

    // set the command type (and the returned data, if needed)
    if (m_Write[ResIdx]) {
      // a write response

#ifdef DEBUG_G1
      PrintTime("Start write response in Slave ",m_ID,m_Address[ResIdx]);
#endif

      // Inform master that this is a Write response
      // To do so, we use the sdatainfo field
      m_DataCl->SputSDataInfo(1);
    } else {
      // a read response and its data

#ifdef DEBUG_G1
      PrintTime("Start read response in Slave ",m_ID,m_Address[ResIdx]);
#endif

      // the to-be-returned read data is stored at *m_Mdata[ResIdx]
      m_DataCl->SputSData(m_Mdata[ResIdx],m_NumWords[ResIdx]);

      // Inform master that this is a Read response
      // To do so, we use the sdatainfo field
      m_DataCl->SputSDataInfo(0);

    }

    // reset the FIFO entry
    m_SlaveRespWait[ResIdx] = true;

    // Send a response to the Master
    if (!SlaveP->SputResponseBlocking()) {
      // Deep ship: Should never happen
      cout << "Error: Master (" << m_BusID << ") dropped Slave (" 
           << m_ID << ") response <" << name() << ">" << endl;
    }

    // get to the next waiting response
    ResIdx++;
    if (ResIdx >= TL_SLAVE_FIFO_DEPTH)
      ResIdx = 0;

    // advance the committed response counter
    m_ResTaNum++;
  }
}

// ----------------------------------------------------------------------------
//  Method : OCP_TL2_Slave::end_of_elaboration()
//
//  This method is activated at the end of the elaboration phase
//  (when all binding operations are finished).
// ----------------------------------------------------------------------------
template<class TdataCl>
void OCP_TL2_Slave<TdataCl>::end_of_elaboration()
{
  //
  sc_module::end_of_elaboration();

  // Initialize debug interface
  SlaveP->SregisterDirectIF(this);

  // Get data structure
  m_DataCl = SlaveP->GetDataCl();

  // Get communication structure
  m_CommCl = SlaveP->GetCommCl();

  // Get system parameter structure
  m_ParamCl = SlaveP->GetParamCl();

  // Put parameter into Channel
  m_ParamCl->SlaveID = m_ID;
  m_ParamCl->StartAddress = m_StartAddress;
  m_ParamCl->EndAddress   = m_EndAddress;
  m_BusID = m_ParamCl->MasterID;
}

// ----------------------------------------------------------------------------
//  Method : OCP_TL2_Slave::MPutDirect()
//
//  Debug interface method.
//  Read/Write data, Master to Slave direction
//  Returns true for success and false for failure
//
// ----------------------------------------------------------------------------
template<class TdataCl>
bool OCP_TL2_Slave<TdataCl>::MputDirect(int MasterID, bool IsWrite, Td* Data,
  Ta Address, int NumWords)
{
  int i;
  int Offset = (Address - m_StartAddress) / m_NumBytesPerWord;

  if ((Offset >= 0) && ((Offset + NumWords) <= m_SmemoryLength)) {
    if (IsWrite) {
      // write request
      for (i = 0; i < NumWords; i++)
        m_SlaveMemory[Offset + i] = Data[i];
    } else {
      // read request
      for (i = 0; i < NumWords; i++)
        Data[i] = m_SlaveMemory[Offset + i];
    }
    return(true);
  } else {
    cout << "Error: Address out of range at Slave " << m_ID << endl
         << "       Address=" << Address << "  StartAddr=" << m_StartAddress
         << "  _EndAddr=" << m_EndAddress
         << "  NumWords=" << NumWords
         << endl;
    return(false);
  }
}

// ----------------------------------------------------------------------------
//  Method : OCP_TL2_Slave::PrintTrans()
//
//  Print transaction
//
// ----------------------------------------------------------------------------
template<class TdataCl>
void OCP_TL2_Slave<TdataCl>::PrintTrans(int TaNum, Ta Addr, int NumWords,
  const char* Method)
{
  double ttime = sc_time_stamp().to_double();
  cout << "Slave " << m_ID << " : " << "Req = " << TaNum
       << "  Address = " << Addr 
       << "  NumWords = " << NumWords << Method 
       << " at "<< ttime << " sec" << endl;
}

// ----------------------------------------------------
//  Method : PrintTime()
//
//  Print time stamp
// ----------------------------------------------------
template<class TdataCl>
void OCP_TL2_Slave<TdataCl>::PrintTime(const char* Pstring, int ID, Ta Addr)
{
  cout << Pstring << ID << " : Address = " << Addr << " at "
       << sc_time_stamp().to_double()
       << endl;
}

// ----------------------------------------------------
//  Method : PrintFuTime()
//
//  Print future time stamp
// ----------------------------------------------------
template<class TdataCl>
void OCP_TL2_Slave<TdataCl>::PrintFuTime(const char* Pstring, int ID, Ta Addr,
  double AddTime)
{
  cout << Pstring << ID << " : Address = " << Addr << " at "
       << (sc_time_stamp().to_double() + AddTime)
       << endl;
}

// ----------------------------------------------------------------------------
//
//  Instantiation of the Slave
//
// ----------------------------------------------------------------------------

#include "ocp_tl2_data_cl.h"
template class OCP_TL2_Slave< OCP_TL2_TEMPL_DATA_CL >;
