///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// (c) Copyright OCP-IP 2008
// OCP-IP Confidential and Proprietary
//
//
//============================================================================
//      Project : OCP SLD WG
//       Author : James Aldis, Texas Instruments
//                Robert Guenzel (from TU of Braunschweig) for Greensocs Ltd.
//
//          $Id:
//
//  Description :  Merger module for the TL1 timing example
//
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#ifndef __TIMING_MERGER_H__
#define __TIMING_MERGER_H__


#include "timing_common.h"

struct pseudo_thread_ID : public tlm_utils::instance_specific_extension<pseudo_thread_ID>{
};

class timing_merger :
  public sc_core::sc_module
{
  SC_HAS_PROCESS(timing_merger);

  tlm_utils::instance_specific_extension_accessor acc;
  public:
  // ports
  sc_core::sc_in<bool> clk;
  ocpip::ocp_slave_socket_tl1<>  ocpsA;
  ocpip::ocp_slave_socket_tl1<>  ocpsB;
  ocpip::ocp_master_socket_tl1<> ocpm;


    timing_merger(sc_core::sc_module_name name, int threads_Ap, int threads_Bp)
      : sc_core::sc_module(name)
      , ocpsA("ocpsA", this, &timing_merger::setOCPTL1MasterTiming<0>)
      , ocpsB("ocpsB", this, &timing_merger::setOCPTL1MasterTiming<1>)
      , ocpm("ocpm", this, &timing_merger::setOCPTL1SlaveTiming)
      , threads_A(threads_Ap)
      , threads_B(threads_Bp)
      , mtb_A(0)
      , mtb_B(0)
      , arb_event_notified(false) {

      ocpsA.register_nb_transport_fw(this, &timing_merger::nb_transport_fwA);
      ocpsB.register_nb_transport_fw(this, &timing_merger::nb_transport_fwB);
      ocpm.register_nb_transport_bw(this, &timing_merger::nb_transport_bw);


      SC_METHOD(clock_rising);
      sensitive<<clk.pos();
      dont_initialize();

      SC_METHOD(request_arb);
      sensitive<<arbiter_event;
      dont_initialize();

      typedef tlm::tlm_generic_payload* tlm_generic_payload_ptr;
      req_reg = //new arbitration_signal<tlm::tlm_generic_payload*>[threads_A + threads_B];
                new tlm_generic_payload_ptr[threads_A + threads_B];
      for (int i=0; i<threads_A + threads_B; i++) req_reg[i]=NULL;
      reqA_reg = req_reg;
      reqB_reg = &(req_reg[threads_A]);

      tb_txn_A=ocpsA.get_tb_transaction();
      tb_txn_B=ocpsB.get_tb_transaction();
      tb_txn_m=ocpm.get_tb_transaction();
      tb_ph_s=ocpip::CMD_THREAD_BUSY_CHANGE;
      tb_ph_m=ocpip::RESP_THREAD_BUSY_CHANGE;

      std::cout << "<<<< E-O-E >>>> " << sc_core::sc_module::name() << std::endl;
      time_quantum = sc_core::sc_get_time_resolution();
      update_timing();

      map_string_type config_map=get_config_map(threads_Ap,32);
      ocpip::ocp_parameters  config;
      config=create_ocp_configuration_from_map(ocpsA.name(), config_map);
      ocpsA.set_ocp_config(config);

      config_map=get_config_map(threads_Bp, 32);
      config=create_ocp_configuration_from_map(ocpsB.name(), config_map);
      ocpsB.set_ocp_config(config);

      config_map=get_config_map(threads_Bp+threads_Ap, 32);
      config=create_ocp_configuration_from_map(ocpm.name(), config_map);
      ocpm.set_ocp_config(config);

      tb_A=ocpsA.get_extension<ocpip::cmd_thread_busy>(*tb_txn_A);
      tb_B=ocpsB.get_extension<ocpip::cmd_thread_busy>(*tb_txn_B);
      tb_m=ocpm.get_extension<ocpip::resp_thread_busy>(*tb_txn_m);
    };

    ~timing_merger() {
      std::cout << "Deleting merger:   " << name() << std::endl;

      for(int i=0; i<threads_A; i++) {
        std::cout << (reqA_reg[i] ? "A " : "- ");
      }
      std::cout << std::endl;
      for(int i=0; i<threads_B; i++) {
        std::cout << (reqB_reg[i] ? "B " : "- ");
      }
      std::cout << std::endl;
      std::cout << "  SThreadBusy sample time: " <<
        time_quantum + ocpm_timing.SThreadBusyStartTime << std::endl;
      std::cout << "  MThreadBusy A sample time: " <<
        ocps_timing[0].MThreadBusyStartTime << std::endl;
      std::cout << "  MThreadBusy B sample time: " <<
        ocps_timing[1].MThreadBusyStartTime << std::endl;
      std::cout << "  Request A sample time:     " <<
        time_quantum + ocps_timing[0].RequestGrpStartTime << std::endl;
      std::cout << "  Request B sample time:     " <<
        time_quantum + ocps_timing[1].RequestGrpStartTime << std::endl;
      std::cout << "  Response sample time:     " <<
        ocpm_timing.ResponseGrpStartTime << std::endl;
    }

    // processes
    void clock_rising() {

      // generate SThreadBusy for both slave ports based on
      // loading of request registers
      int stb_out = 0;
      int mask = 1;
      for(int i=0; i<(threads_A+threads_B); i++) {
        if(req_reg[i]) {
          stb_out |= mask;
        }
        mask <<= 1;
      }
      tb_A->value=(stb_out & ((1 << threads_A)-1));
      tb_B->value=(stb_out >> threads_A);
      ocpsA->nb_transport_bw(*tb_txn_A, tb_ph_s, tb_time);
      ocpsB->nb_transport_bw(*tb_txn_B, tb_ph_s, tb_time);

      arbiter_event.notify(arb_time);
      arb_event_notified = true;

      // the response path is really simple - thread-busy-signals are
      // concatencated and passed to slave.  slave returns a response which
      // is forwarded to the right slave port.  in both cases there is
      // no process, just the nb_transport()
    }

    // called by nb_transport_fw()
    void mthreadbusy_propagate() {
      tb_m->value=(mtb_A | (mtb_B << threads_A));
      ocpm->nb_transport_fw(*tb_txn_m, tb_ph_m, tb_time);
    }

    // process, triggered a certain time into the clock cycle
    void request_arb() {
      arb_event_notified = false;
      // now arbitrate: priority is smallest thread ID, A then B
      int mask = 1;
      for(int i=0; i<(threads_A+threads_B); i++) {
        if((req_reg[i]) && !(mask & stb)) {
          // grant this one
          txn_ph=tlm::BEGIN_REQ;
          txn_time=sc_core::SC_ZERO_TIME;
          tlm::tlm_sync_enum retVal=ocpm->nb_transport_fw(*req_reg[i], txn_ph, txn_time);
          switch (retVal){
            case tlm::TLM_UPDATED:
              if (txn_ph==tlm::END_REQ); //great! just an immediate accept!
              else{
                std::cerr<<"Unexpected phase "<<txn_ph<<" on return path. In "<<name()<<" at "<<sc_core::sc_time_stamp()<<std::endl;
                exit(1);
              }
              break;
            case tlm::TLM_COMPLETED:
              break;
            default:
              std::cerr<<"When using thread busy exact, I expect the use of the return path. In "<<name()<<" at "<<sc_core::sc_time_stamp()<<std::endl;
              exit(1);
          }
          req_reg[i]=NULL;
          break;
        }
        mask <<= 1;
      }
    }

    // when informed of master port timing, merger must re-inform the OCP
    // channels if anything changed
    void setOCPTL1SlaveTiming(ocpip::ocp_tl1_slave_timing slave_timing) {
      std::cout << "  << S-S-T >>   " << name() << std::endl;
      ocpm_timing = slave_timing;
      update_timing();
    }

    // when informed of master timing, merger must re-inform the OCP
    // channels if anything changed
    template<int PORT>
    void setOCPTL1MasterTiming(ocpip::ocp_tl1_master_timing master_timing) {
      std::cout << "  << S-M-T >>   " << name() << std::endl;
      ocps_timing[PORT] = master_timing;
      update_timing();
    }

    tlm::tlm_sync_enum nb_transport_fwA(tlm::tlm_generic_payload& gp, tlm::tlm_phase& ph, sc_core::sc_time& time){
      if (ph==ocpip::RESP_THREAD_BUSY_CHANGE){
        mtb_A=ocpsA.get_extension<ocpip::resp_thread_busy>(gp)->value;
        mthreadbusy_propagate();
        return tlm::TLM_ACCEPTED;
      }
      if (ph!=tlm::BEGIN_REQ){
        std::cerr<<"I only expect BEGIN_REQ on the forward path, but got "<<ph<<" In "<<name()<<" at "<<sc_core::sc_time_stamp()<<std::endl;
        exit(1);
      }
      ocpip::thread_id* tmp;
      bool thread_id_available=ocpm.get_extension<ocpip::thread_id>(tmp, gp);
      if (!thread_id_available){
        assert(threads_A==1); //if there were more than 1 thread, there should have been a thread ID
        ocpm.validate_extension<ocpip::thread_id>(gp);
        tmp->value=0; //we need a thread ID so we add it and activate it
      }

      assert(!(((tb_A->value)>>(tmp->value)) & 1)); //make sure our thread is not busy

      //assert that we have not seen this thing before
      pseudo_thread_ID* check;
      acc(gp).get_extension(check);
      assert(!check);

      acc(gp).set_extension(&threadIDA);
      reqA_reg[tmp->value]=&gp; //at this time the thread id is correct
      ph=tlm::END_REQ;
      return tlm::TLM_UPDATED;
    }

    tlm::tlm_sync_enum nb_transport_fwB(tlm::tlm_generic_payload& gp, tlm::tlm_phase& ph, sc_core::sc_time& tim){
      if (ph==ocpip::RESP_THREAD_BUSY_CHANGE){
        mtb_B=ocpsB.get_extension<ocpip::resp_thread_busy>(gp)->value;
        mthreadbusy_propagate();
        return tlm::TLM_ACCEPTED;
      }
      if (ph!=tlm::BEGIN_REQ){
        std::cerr<<"I only expect BEGIN_REQ on the forward path, but got "<<ph<<" In "<<name()<<" at "<<sc_core::sc_time_stamp()<<std::endl;
        exit(1);
      }
      ocpip::thread_id* tmp;
      bool thread_id_available=ocpm.get_extension<ocpip::thread_id>(tmp, gp);
      if (!thread_id_available){
        assert(threads_B==1); //if there were more than 1 thread, there should have been a thread ID
        ocpm.validate_extension<ocpip::thread_id>(gp); //NOTE: we use the multi threaded socket to set the extension, because the single threaded sockets
                                                // do not know the extension!
        tmp->value=0; //we need a thread ID so we add it and activate it
      }

      assert(!(((tb_B->value)>>(tmp->value)) & 1)); //make sure our thread is not busy


      //assert that we have not seen this thing before
      pseudo_thread_ID* check;
      acc(gp).get_extension(check);
      assert(!check);

      acc(gp).set_extension(&threadIDB);

      reqB_reg[tmp->value]=&gp; //at this time the thread id is correct

      tmp->value+=threads_A; //change threadID
      ph=tlm::END_REQ;
      return tlm::TLM_UPDATED;
    }

    tlm::tlm_sync_enum nb_transport_bw(tlm::tlm_generic_payload& gp, tlm::tlm_phase& ph, sc_core::sc_time& t){
      if (ph==ocpip::CMD_THREAD_BUSY_CHANGE){
        stb=ocpm.get_extension<ocpip::cmd_thread_busy>(gp)->value;
        return tlm::TLM_ACCEPTED;
      }

      if (ph!=tlm::BEGIN_RESP){
        std::cerr<<"I only expect BEGIN_RESP on the backward path, but got "<<ph<<" In "<<name()<<" at "<<sc_core::sc_time_stamp()<<std::endl;
        exit(1);
      }
      pseudo_thread_ID* threadID;
      acc(gp).get_extension(threadID);
      acc(gp).clear_extension(threadID); //remove the thing from the txn
      if (threadID==&threadIDA)
        return ocpsA->nb_transport_bw(gp, ph, t);
      else
      if (threadID==&threadIDB)
        return ocpsB->nb_transport_bw(gp, ph, t);
      else{
        std::cerr<<"There is a response which was not tagged with a local thread ID. That's strange... In "
                 <<name()<<" at "<<sc_core::sc_time_stamp()<<std::endl;
        exit(1);
      }
      return tlm::TLM_ACCEPTED;
    }

  private:
    // store what we have been told by the other modules
    ocpip::ocp_tl1_master_timing ocps_timing[2];
    ocpip::ocp_tl1_slave_timing ocpm_timing;

    // times used internally - these are the output times
    sc_core::sc_time time_quantum;
    sc_core::sc_time resp_time;
    sc_core::sc_time mtb_time;
    sc_core::sc_time arb_time;

    sc_core::sc_time calc_resp_time() {
      return ocpm_timing.ResponseGrpStartTime;
    }
    sc_core::sc_time calc_mtb_time() {
      return my_max(
        ocps_timing[0].MThreadBusyStartTime,
        ocps_timing[1].MThreadBusyStartTime);
    }
    sc_core::sc_time calc_arb_time() {
      return time_quantum + my_max(
        ocpm_timing.SThreadBusyStartTime, my_max(
        ocps_timing[0].RequestGrpStartTime,
        ocps_timing[1].RequestGrpStartTime));
    }

    // a function to look at the timing and update if necessary
    // only called at cycle start or during elaboration
    void update_timing() {
      sc_core::sc_time new_time = calc_resp_time();
      if(new_time != resp_time) {
        resp_time = new_time;
        ocpip::ocp_tl1_slave_timing st;
        st.ResponseGrpStartTime = resp_time;
        ocpsA.set_slave_timing(st);
        ocpsB.set_slave_timing(st);
      }
      new_time = calc_mtb_time();
      sc_core::sc_time new_time0 = calc_arb_time();
      if((new_time0 != arb_time) || (new_time != mtb_time)) {
        arb_time = new_time0;
        mtb_time = new_time;
        ocpip::ocp_tl1_master_timing mt;
        mt.RequestGrpStartTime = arb_time;
        mt.MThreadBusyStartTime = mtb_time;
        ocpm.set_master_timing(mt);
        if(arb_event_notified) {
          arbiter_event.cancel();
          arbiter_event.notify(arb_time);
        }
      }
    }

    sc_core::sc_event arbiter_event, mthreadbusy_event;
    int threads_A, threads_B, mtb_A, mtb_B, stb;
    pseudo_thread_ID threadIDA, threadIDB;
    tlm::tlm_generic_payload *tb_txn_A, *tb_txn_B, *tb_txn_m, **req_reg, **reqA_reg, **reqB_reg;
    tlm::tlm_phase tb_ph_s, tb_ph_m, txn_ph;
    ocpip::cmd_thread_busy *tb_A, *tb_B;
    ocpip::resp_thread_busy *tb_m;
    sc_core::sc_time tb_time, txn_time;
    bool arb_event_notified;
};


// end of multiple inclusion protection
#endif

