// 
//  Copyright 2008 OCP-IP
//
// ============================================================================
//      Project : OCP SLD WG
//       Author : Anssi Haverinen, Nokia Inc.
//                Robert Guenzel (from TU of Braunschweig) for Greensocs Ltd.
//         $Id: slave.cc,v 1.2 2004/09/05 17:57:56 halexan Exp $
//
//  Description : OCP API - Layer-1 Simple Example slave
//    Features:
//    - Synchronous
//    - Non-blocking methods are used
//    - Not pipelined
// ============================================================================

#include "slave.h"

// ----------------------------------------------------------------------------
// Constructor
// ----------------------------------------------------------------------------
template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH> Slave<BUSWIDTH, ADDRWIDTH>::Slave (sc_core::sc_module_name name_)
  : sc_core::sc_module (name_), tpP("tpPort"), req(NULL), ongoing_rsp(false) {

  // initialize common members
  last_request = tlm::TLM_IGNORE_COMMAND;
  dataCnt = 0;

  SC_METHOD(proc);
  sensitive<<clk.pos();
  dont_initialize(); // make sure we execute only at clock edge
  tpP.register_nb_transport_fw(this, &Slave::nb_transport);
  tpP.activate_synchronization_protection();
}


// ----------------------------------------------------------------------------
// Destructor
// ----------------------------------------------------------------------------
template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH> Slave<BUSWIDTH, ADDRWIDTH>::~Slave(){}


// ----------------------------------------------------------------------------
//  Method : Slave::proc()
//
//  Synchronous slave process
//
// ----------------------------------------------------------------------------

template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH> void Slave<BUSWIDTH, ADDRWIDTH>::proc(){

  // If a previous read has been served, we check for new requests
  if (last_request == tlm::TLM_IGNORE_COMMAND) {
    if (req){
      last_request = req->get_command();

      if (last_request == tlm::TLM_READ_COMMAND)
#ifdef DEBUG_G1                  
      {
#endif
        rsp=req;
#ifdef DEBUG_G1                  
        std::cout << "Slave got read request "
                  << "  time  = " << sc_core::sc_time_stamp().to_seconds()
                  << "  address  = " << rsp->get_address() << std::endl;
      }
      else{
        std::cout << "Slave got write request "
                  << "  time  = " << sc_core::sc_time_stamp().to_seconds()
                  << "  data  = " << (*((Td*)(req->get_data_ptr()))) << std::endl;

      }
#endif
      if (last_request == tlm::TLM_WRITE_COMMAND) {
        assert(tpP.template get_extension<ocpip::posted>(*req)); //we do not support nonposted writes
      }
     
      time=sc_core::SC_ZERO_TIME;
      phase=tlm::END_REQ;
      tlm::tlm_sync_enum retVal=tpP->nb_transport_bw(*req,phase, time);
      switch(retVal){
        case tlm::TLM_ACCEPTED: break;
        case tlm::TLM_UPDATED:
          std::cerr<<"Error in "<<name()<<" got unexpected phase update "<<phase<<" in response to END_REQ"<<std::endl;
          exit(1);
          break;
        case tlm::TLM_COMPLETED:;
      }      
      req=NULL;
    }
  }

  // If request was write
  if (last_request == tlm::TLM_WRITE_COMMAND) {
    last_request = tlm::TLM_IGNORE_COMMAND;
  }
  else 
  if (last_request == tlm::TLM_READ_COMMAND) {	
    if (!ongoing_rsp) {
      assert(rsp);
      *((Td*)(rsp->get_data_ptr())) = dataCnt++;
      rsp->set_response_status(tlm::TLM_OK_RESPONSE);
      // Send response
      time=sc_core::SC_ZERO_TIME;
      phase=tlm::BEGIN_RESP;
      tlm::tlm_sync_enum retVal=tpP->nb_transport_bw(*rsp,phase, time);
      switch(retVal){
        case tlm::TLM_ACCEPTED: ongoing_rsp=true; break;
        case tlm::TLM_UPDATED:
          if (phase!=tlm::END_RESP) {
            std::cerr<<"Error in "<<name()<<" got unexpected phase update "<<phase<<" in response to BEGIN_RESP"<<std::endl;
            exit(1);
          }
          break;
        case tlm::TLM_COMPLETED: break;
      }

#ifdef DEBUG_G1
      std::cout << "Slave sent response " << rsp->get_response_status()
	   << " time " << sc_core::sc_time_stamp().to_seconds()
	   << " data " << (*(((Td*)(rsp->get_data_ptr()))+reqcnt-1)) << std::endl;
#endif
      last_request = tlm::TLM_IGNORE_COMMAND;
      rsp=NULL;
    }
  } // end if (last_request)
} // end of method

template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH>
tlm::tlm_sync_enum Slave<BUSWIDTH, ADDRWIDTH>::nb_transport(tlm::tlm_generic_payload& txn, tlm::tlm_phase& ph, sc_core::sc_time& tim){
  switch(ph){
    case tlm::BEGIN_REQ: req=&txn; break;
    case tlm::END_RESP: ongoing_rsp=false; break;
    default:
      std::cerr<<"Error in "<<name()<<" got unexpected phase "<<phase<<" in nb_transport_fw"<<std::endl;
      exit(1);        
  }
  return tlm::TLM_ACCEPTED;
}

// ----------------------------------------------------------------------------
//
//  Instantiation of the Slave
//
// ----------------------------------------------------------------------------
template class Slave< 32,32 >; 

