// 
//  (c) Copyright OCP-IP 2003, 2004
//  OCP-IP Confidential and Proprietary
//
// ============================================================================
//      Project : OCP SLD WG, Layer adapter examples 
//      Authors : Stephane Guntz, PROSILOG, guntz@prosilog.com
//				  Yann Bajot, PROSILOG, bajot@prosilog.com               
//
//  Description : Transaction Level - Layer-1 to Layer-2 Master Adapter (OCP 2.0 supported)  
//  $Id: ocp_tl1_tl2_master_adapter.cpp,v 1.1.1.1 2004/09/24 09:26:27 sguntz Exp $
//  
//  Modifications (09/01/2004): 
//		- correction of array indexing,
//		- correct bug for computing of address_increment and making relevant TL2 request chunks (occured when master adapter receives more than adapter_depth TL2 chunks)
//		- correct bug when a response chunk is being processed, the buffer is full and a new one is received,
//		- add master adapter name in debug printing messages
//		- change some debug messages
//
//  Modifications (09/22/2004): new version which is compatible with both new TL2 performance channel and old layered TL2 channel
//	    - remove TL2 dataclass as a template (only one template now: the TL1 dataclass)
//	    - remove references to direct interfaces 
//	
//  Parameters    :
//    Template arguments.
//     - TdataCl_tl1: TL1 Data class containing data members and
//                access methods for the data exchanged between
//                Masters and Slaves
//	                   
//	Constructor arguments:
//      - sc_module_name: Name of the module instance
//	    - max_chunk_length: maximum size of a chunk
//		- adapter_depth: Number of missed events and responses to be cached.
//						 The adapter stores new request/response infos which have arrived during the processing of the current request/response
//						 and sends them when the current request/response is finished
//		- timeout: a warning is sent if during a timeout period, no TL1 request or
//		TL2 response has been received by the master adapter 
//	 
//	 
//  The TL1 master adapter has a TL2 master interface on one side, and a TL1 slave interface on the other side.
//  It must be connected to:
//      - a TL1 master through a TL1 channel
//      - a TL2 slave through a TL2 channel
//
//	The adapter retrieves TL1 requests and stores them in its internal buffer. When a new request arrived, the adapter
//	compares it with former requests to decide whether it is part of the current burst or the beginning of a new burst
//  (it can be a single request too).
//  When a burst is finished, a TL2 request is sent to the slave.
//  
//  On the response side, the TL1 master adapter retrieves the TL2 response from the slave, and sends corresponding
//  TL1 responses to the master. 
// 
//	The following is unsupported:
//	- Single request/multiple data
//	- Data handshake
//	- Multiple threads
//	- SThreadBusy/MThreadBusy compliances (m_threadbusy_exact=0, s_threadbusy_exact=0)
// 
// ============================================================================

#include "ocp_tl1_tl2_master_adapter.h"


//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter constructor
//---------------------------------------------------------------------------
template<class TdataCl_tl1>
OCP_TL1_TL2_Master_Adapter<TdataCl_tl1>::OCP_TL1_TL2_Master_Adapter(sc_module_name name_, int max_chunk_length_, int adapter_depth_, sc_time timeout): 
	sc_module(name_),
	SlaveP("SlaveP"),
	MasterP("MasterP"),
    max_chunk_length(max_chunk_length_),
	adapter_depth(adapter_depth_),
	m_timeout(timeout)
{

	buffer_size=max_chunk_length*adapter_depth;

	m_request_buffer=new OCPRequestGrp<Td, Ta>[buffer_size];
	m_response_buffer= new OCPResponseGrp<Td>[adapter_depth];
	m_data=new Td[buffer_size];
	m_response_data= new Td[buffer_size];

	//arrays for storing characteristics of TL2 requests
	burst_request_index=new int[adapter_depth];
	burst_request_length=new int[adapter_depth];

	//arrays for storing characteristics of TL1 responses
	response_index=new int[adapter_depth];
	response_length=new unsigned int[adapter_depth];

	//initialize internal request, data and response buffer
	for(int i=0; i<buffer_size; i++)  {
		reset_request_parameters(i);
		m_data[i]=0;
		m_response_data[i]=0;
	}

	beginning_burst_index= current_index=0;

	for(int i=0; i<adapter_depth; i++)   {
		burst_request_index[i]=0;
		burst_request_length[i]=0;
		response_index[i]=0;
		response_length[i]=0;
	}
	new_burst_transaction=true;
	new_chunk_transaction=true;

	last_of_a_burst=false;
	release_request_thread=false;
	release_response_thread=false;
	request_not_finished=false;
    m_pending_request=0;
	m_request_number=0;
	m_pending_response=0;
	m_response_number=0;
	current_processed_request=current_processed_response=0;

	m_NumBytesPerWord = sizeof(Td);

	//SystemC processes 
	SC_THREAD(SlaveRequest);
	sensitive<<SlaveP.RequestStartEvent();

	SC_THREAD(MasterRequest);
	sensitive<<send_TL2_request_event;

	SC_THREAD(SlaveResponse);
    sensitive<<MasterP.ResponseStartEvent();

	SC_THREAD(MasterResponse);
	sensitive<<send_TL1_response_event;

	SC_METHOD(timeout_activated);
	dont_initialize();
	sensitive<<timeout_event;


}

//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter destructor
//---------------------------------------------------------------------------
template<class TdataCl_tl1>
OCP_TL1_TL2_Master_Adapter<TdataCl_tl1>::~OCP_TL1_TL2_Master_Adapter()    {
	
	if(m_request_buffer)  {
		delete [] m_request_buffer;
		m_request_buffer=0;
	}
	if(m_response_buffer)  {
		delete [] m_response_buffer;
		m_response_buffer=0;
	}
	if(m_response_data)  {
		delete [] m_response_data;
		m_response_data=0;
	}
	if(m_data)  {
		delete [] m_data;
		m_data=0;
	}
	if(burst_request_index)  {
		delete [] burst_request_index;
		burst_request_index=0;
	}
	if(burst_request_length)  {
		delete [] burst_request_length;
		burst_request_length=0;
	}
	if(response_index)  {
		delete [] response_index;
		response_index=0;
	}
	if(response_length)  {
		delete [] response_length;
		response_length=0;
	}
	if(burst_precise && precise_burst_length )  {
		delete [] precise_burst_length;
		precise_burst_length=0;
	}
}

//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter::SlaveRequest 
//get the TL1 requests, store them in the internal buffer, prepare TL2 request to be sent to the TL2 slave
//---------------------------------------------------------------------------
template<class TdataCl_tl1>
void OCP_TL1_TL2_Master_Adapter<TdataCl_tl1>::SlaveRequest()   {
	OCPRequestGrp<Td,Ta> req;

	while(true)   {
		wait();			//wait for default event of the channel
		
        if(SlaveP->getOCPRequest(req, false))   {  //get a pending request

			//a request has arrived: cancel the previous timeout notification 
			timeout_event.cancel();
			//notify the timeout event 
			timeout_event.notify(m_timeout);

			//store the request fields in the internal buffer
			m_request_buffer[current_index].copy(req);

			//store the data in the internal buffer
		    m_data[current_index]=req.MData;

			#ifdef DEBUG_MASTER_ADAPTER_TL1   
				cout<<"---\nTL1-TL2 Master Adapter \""<<name()<< "\" :";
				if(new_burst_transaction)
					cout<<"\nSlaveRequest(): master adapter gets the first TL1 request of a burst at time "<<sc_time_stamp()<<endl;
				else cout<<"\nSlaveRequest(): master adapter gets a TL1 request at time "<<sc_time_stamp()<<endl;
				cout<<"MAddr: "<<m_request_buffer[current_index].MAddr<<endl;
				cout<<"MBurstSeq: "<<m_request_buffer[current_index].MBurstSeq<<endl;
				cout<<"MThreadID: "<<m_request_buffer[current_index].MThreadID<<endl;
				cout<<"MCmd: "<<m_request_buffer[current_index].MCmd<<endl;
			#endif
								
			switch(define_adapter_task())    {

			case SEND_CHUNK_WITH_REQUEST:
				#ifdef DEBUG_MASTER_ADAPTER_TL1  
					cout<<"The current TL2 chunk will be sent WITH this TL1 request"<<"\n---"<<endl;
				#endif 
				
				//specify burst length and beginning
				burst_request_index[m_request_number]=beginning_burst_index;
				
				if(current_index<beginning_burst_index)   
					burst_request_length[m_request_number]=current_index+buffer_size-beginning_burst_index+1; 
				else burst_request_length[m_request_number]=current_index-beginning_burst_index+1; 
			
				//change the internal index
				if(current_index!=buffer_size-1)
					beginning_burst_index=current_index+1;
				else beginning_burst_index=0;

				if(new_burst_transaction)
					last_of_a_burst=true;
				else last_of_a_burst=false;
				
				//increment number of pending requests that the adapter has to send
				m_pending_request++;

				//increment number of the next request that is going to be stored
				//check if the request buffer is full: next request is currently being processed
				if((m_request_number+1)%adapter_depth==current_processed_request)  {
                    release_request_thread=true;
                }
				//check if the response buffer is full
                else if((m_response_number+1)%adapter_depth==current_processed_response)  {
                    request_not_finished=true;
                    m_request_number=(m_request_number+1)%adapter_depth;
                }
				else {
					release_request_thread=false;
					m_request_number=(m_request_number+1)%adapter_depth;
					SlaveP->putSCmdAccept();
				}
				
				//notify request event
				send_TL2_request_event.notify();	
				break;

			case SEND_CHUNK_WITHOUT_REQUEST:
				#ifdef DEBUG_MASTER_ADAPTER_TL1  
					cout<<"The current TL2 chunk will be sent WITHOUT this TL1 request"<<"\n---"<<endl;
				#endif 

				//specify burst length and beginning
				burst_request_index[m_request_number]=beginning_burst_index;
				
				if(current_index<beginning_burst_index)   
					burst_request_length[m_request_number]=current_index+buffer_size-beginning_burst_index; 
				else burst_request_length[m_request_number]=current_index-beginning_burst_index; 
			
				
				//change the internal index
				beginning_burst_index=current_index;

				if(new_burst_transaction)
					last_of_a_burst=true;
				else last_of_a_burst=false;

				//increment number of pending requests that the adapter has to send
				m_pending_request++;
				//increment number of the next request that is going to be stored
				//check if the request buffer is full: next request is not currently being processed
				if((m_request_number+1)%adapter_depth==current_processed_request)
					release_request_thread=true;
                //check if the response buffer is full
                else if((m_response_number+1)%adapter_depth==current_processed_response)  {
                    request_not_finished=true;
                    m_request_number=(m_request_number+1)%adapter_depth;
                }
				else {
					release_request_thread=false;
					m_request_number=(m_request_number+1)%adapter_depth;
					SlaveP->putSCmdAccept();
				}
				
				//notify request event
				send_TL2_request_event.notify();
				break;

			case STORE_NEW_REQUEST:
				#ifdef DEBUG_MASTER_ADAPTER_TL1  
					cout<<"This TL1 request is stored with the current TL2 chunk"<<"\n---"<<endl;
				#endif 
				SlaveP->putSCmdAccept();
				release_request_thread=false;
				break;

			case SEND_CHUNK_AND_SINGLE_REQUEST:
				#ifdef DEBUG_MASTER_ADAPTER_TL1  
					cout<<"The current TL2 chunk will be sent WITHOUT this TL1 request. This request will be sent as a SINGLE TL2 chunk"<<"\n---"<<endl;
				#endif 

				//specify the characteristics of the current burst
				//specify burst length and beginning
				burst_request_index[m_request_number]=beginning_burst_index;
			
				if(current_index<beginning_burst_index)   
					burst_request_length[m_request_number]=current_index+buffer_size-beginning_burst_index; 
				else burst_request_length[m_request_number]=current_index-beginning_burst_index; 
			
				//change the internal index
				beginning_burst_index=current_index;
				
				if(current_index!=buffer_size-1)
					current_index++;
				else current_index=0;

				//increment number of pending requests that the adapter has to send
				m_pending_request++;

				//increment number of the next request that is going to be stored
				//check if the request buffer is full: next request is not currently being processed
				if((m_request_number+1)%adapter_depth==current_processed_request)    {
					release_request_thread=true;
					send_TL2_request_event.notify();
					wait(thread_released_event);  //wait for the current request to be finished to store the characteristics of the next request
				}
				//check if the response buffer is full
                else if((m_response_number+1)%adapter_depth==current_processed_response)  {
                    request_not_finished=true;
                    m_request_number=(m_request_number+1)%adapter_depth;
                }
                else {
					release_request_thread=false;
					m_request_number=(m_request_number+1)%adapter_depth;
				}

				//continue with single request
				burst_request_index[m_request_number]=beginning_burst_index;
				burst_request_length[m_request_number]=1; //use the abs function because of circular buffer

				//change the internal index
				if(current_index!=buffer_size-1)
					beginning_burst_index=current_index+1;
				else beginning_burst_index=0;

				//last datum of the burst is part of the burst
				last_of_a_burst=true;

				//increment number of pending requests that the adapter has to send
				m_pending_request++;

				//increment number of the next request that is going to be stored
				//check if the request buffer is full: next request is not currently being processed
				if((m_request_number+1)%adapter_depth==current_processed_request)  {
					release_request_thread=true;
                }
				//check if the response buffer is full
                else if((m_response_number+1)%adapter_depth==current_processed_response)  {
                    request_not_finished=true;
                    m_request_number=(m_request_number+1)%adapter_depth;
                }
                else {
					release_request_thread=false;
					m_request_number=(m_request_number+1)%adapter_depth;
					SlaveP->putSCmdAccept();
				}
				
				//notify request event
				send_TL2_request_event.notify();
				break;

			default:
				break;
			}		

			//change the internal index
			if(current_index!=buffer_size-1)
				current_index++;
			else current_index=0;

		}  //get a pending request
	} //end of infinite loop
}  //end of SlaveRequest


//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter::MasterRequest
//send TL2 request made of atomic TL1 requests
//
//---------------------------------------------------------------------------
template< class TdataCl_tl1>
void OCP_TL1_TL2_Master_Adapter<TdataCl_tl1>::MasterRequest()   {
	OCPRequestGrp<Td,Ta> req;
	
	Td* tl2_request_data=new Td[max_chunk_length];
	for(int i=0; i<max_chunk_length; i++)
		tl2_request_data[i]=0;

	while(true)   {
		
		if(m_pending_request==0)
			wait();  //wait for notification of send_TL2_request_event

		//put pointer to data and chunk length in the TL2 channel
		//put request fields into the TL2 channel
		req.copy(m_request_buffer[burst_request_index[current_processed_request]]);

		//set the request data
		for(int i=0; i<burst_request_length[current_processed_request]; i++)
			tl2_request_data[i]=m_data[(i+burst_request_index[current_processed_request])%buffer_size];

		req.MDataPtr=&tl2_request_data[0];

		
		#ifdef DEBUG_MASTER_ADAPTER_TL1
			switch(req.MCmd)   {
				case OCP_MCMD_WR:
					cout<<"\n*****\nTL1-TL2 Master Adapter \""<<name()<< "\" :";
					cout<<"\nMasterRequest(): sends a TL2 write request at time t="<<sc_time_stamp()<<endl;
					break;
				case OCP_MCMD_RD:
					cout<<"\n*****\nTL1-TL2 Master Adapter \""<<name()<< "\" :";
					cout<<"\nMasterRequest(): sends a TL2 read request at time t="<<sc_time_stamp()<<endl;
					break;
				case OCP_MCMD_RDEX:
					cout<<"\n*****\nTL1-TL2 Master Adapter \""<<name()<< "\" :";
					cout<<"\nMasterRequest(): sends an exclusive TL2 read request at time t="<<sc_time_stamp()<<endl;
					break;
				default:
					cout<<"\n++++TL1-TL2 Master Adapter \""<<name()<< "\" :";
					cout<<" MasterRequest(): default case \n++++"<<endl;
					break;
			} //end of switch
			cout<<"Length: "<< burst_request_length[current_processed_request]<<endl;
			if(last_of_a_burst)
				cout<<"Last chunk of the burst\n*****"<<endl;
			else cout<<"Not the last chunk of the burst\n*****\n"<<endl;
		#endif
			

		//send TL2 request to slave
		MasterP->sendOCPRequestBlocking(req, burst_request_length[current_processed_request], last_of_a_burst);
	
		if(request_not_finished )  {
            request_not_finished=false;
            SlaveP->putSCmdAccept();
        }  

		//processing next TL2 request
        current_processed_request=(current_processed_request+1)%adapter_depth;

		//release the request thread, if it was not released before
		if(release_request_thread)  {
			m_request_number=(m_request_number+1)%adapter_depth;
			SlaveP->putSCmdAccept();
			release_request_thread=false;
		}
				
		if(m_pending_request!=0)
			m_pending_request--;

	} //end of infinite loop

} //end of MasterRequest method



//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter::SlaveResponse
//get the TL2 response from the slave
//---------------------------------------------------------------------------
template<class TdataCl_tl1>
void OCP_TL1_TL2_Master_Adapter<TdataCl_tl1>::SlaveResponse()   {

	unsigned int i=0;
	OCPResponseGrp<Td> resp;

	bool last_chunk=false, first_response=true;
	unsigned int resp_length=0;

	while(true)    {
		//get TL2 response from the slave 
        if(MasterP->getOCPResponseBlocking(resp, false, resp_length, last_chunk))   {
			
			response_length[m_response_number]=resp_length;

			//a response has arrived: cancel the previous timeout notification 
			timeout_event.cancel();
			//notify the timeout event 
			timeout_event.notify(m_timeout);

			#ifdef DEBUG_MASTER_ADAPTER_TL1
				cout<<"\n---\nTL1-TL2 Master Adapter \""<<name()<< "\" : ";
				cout<<"Slave has sent a TL2 response at time t="<<sc_time_stamp()<<endl;
			#endif

			//copy response fields from TL2 data class to adapter
            m_response_buffer[m_response_number].copy(resp);

			//response
			#ifdef DEBUG_MASTER_ADAPTER_TL1
				cout<<"TL2 response:"<<endl;
				cout<<"SResp= "<<m_response_buffer[m_response_number].SResp<<endl;
				cout<<"SThreadID= "<<m_response_buffer[m_response_number].SThreadID<<endl;
				cout<<"Read data= ";
			#endif
			
			//specify the beginning of the current chunk
            if(m_response_number==0)  {
				//if this is the first received TL2 response, set the beginning index to 0
				if(first_response )  {
					response_index[m_response_number]=0;
					first_response=false;
				}
				//or set the beginning index to the next index after the last response
				else response_index[0]=(response_index[adapter_depth-1]+response_length[adapter_depth-1])%buffer_size;		
			}
			else response_index[m_response_number]=(response_index[m_response_number-1]+response_length[m_response_number-1])%buffer_size;

			
			for(i=0; i<response_length[m_response_number]; i++)   {
				m_response_data[(i+response_index[m_response_number])%buffer_size] =resp.SDataPtr[i];
				#ifdef DEBUG_MASTER_ADAPTER_TL1
					cout<<resp.SDataPtr[i] <<" ";
				#endif
			}
			

			#ifdef DEBUG_MASTER_ADAPTER_TL1
				cout<<"\n---"<<endl;
			#endif

			send_TL1_response_event.notify();

			//increment number of pending responses that the adapter has to send
			m_pending_response++;

			//increment next response number
			if((m_response_number+1)%adapter_depth==current_processed_response)  {
				m_response_number=(m_response_number+1)%adapter_depth;
				release_response_thread=true;
			}
			else {
				release_response_thread=false;
				m_response_number=(m_response_number+1)%adapter_depth;
				//release response channel
				MasterP->putMRespAccept(); 
			}
			
		}  //end of get Response

        wait();
	} //end of infinite loop
} //end of SlaveResponse method


//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter::MasterResponse
//send the TL1 response to the master, made from the TL2 response from the slave 
//---------------------------------------------------------------------------
template<class TdataCl_tl1>
void OCP_TL1_TL2_Master_Adapter<TdataCl_tl1>::MasterResponse()   {

	int i= 0;
	OCPResponseGrp<Td> resp;

	while(true)    {
		if(m_pending_response==0)
			wait();  //wait for notification of send_TL1_response_event

	    wait(clk->posedge_event());

        while(SlaveP->SgetMbusy())  
			wait(clk->posedge_event());

		//copies response fields into TL1 data class
		resp.copy(m_response_buffer[current_processed_response]);
        
		//send atomic TL1 responses
		for(i=response_index[current_processed_response]; i<response_index[current_processed_response]+(int)response_length[current_processed_response]; i++)  {
            
            resp.SData=m_response_data[i%buffer_size];

			#ifdef DEBUG_MASTER_ADAPTER_TL1
				cout<<"***\nTL1-TL2 Master Adapter \""<<name()<< "\" : ";
				if(i==response_index[current_processed_response])
					cout<<"sends first TL1 response of a chunk at time= "<<sc_time_stamp()<<endl;
				else if(i==(response_index[current_processed_response]+(int)response_length[current_processed_response]-1))
					cout<<"sends last TL1 response of a chunk at time= "<<sc_time_stamp()<<endl;
				else cout<<"sends TL1 response at time= "<<sc_time_stamp()<<endl;
				cout<<"SData= "<<m_response_data[i%buffer_size]<<"\n***"<<endl;
			#endif

            //send response
			SlaveP->startOCPResponse(resp);

            while(SlaveP->SgetMbusy())  
                wait(clk->posedge_event());

		}
		#ifdef DEBUG_MASTER_ADAPTER_TL1
			cout<<"\n-----------------------------------------\nTL1-TL2 Master Adapter \""<<name()<< "\" :";
			cout<<"\nProcessing next TL1 responses at t= "<<sc_time_stamp();
			cout<<"\n-----------------------------------------"<<endl;
		#endif

		current_processed_response=(current_processed_response+1)%adapter_depth;

		//release response_thread if it not was done before
		if(release_response_thread)    {
			MasterP->putMRespAccept();
            release_response_thread=false;
            thread_released_event.notify();
		}

		if(m_pending_response!=0)
			m_pending_response--;

	} //end of infinite loop
} //end of MasterResponse method



//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter::define_adapter_task
//
//---------------------------------------------------------------------------
template< class TdataCl_tl1>
MasterAdapterTask OCP_TL1_TL2_Master_Adapter<TdataCl_tl1>::define_adapter_task()   {
	bool same_fields=same_request_fields(m_request_buffer[current_index], m_request_buffer[beginning_burst_index]);
	bool same_other_fields=same_other_request_fields(m_request_buffer[current_index], m_request_buffer[beginning_burst_index]);
	bool single=false;

	//completed chunk has maximum length
	if( ( (current_index<beginning_burst_index)  && (current_index+buffer_size-beginning_burst_index+1==max_chunk_length) )
		|| ( (current_index>=beginning_burst_index) && (current_index-beginning_burst_index+1==max_chunk_length) )  ) {
		//the request fields are the same, and address increment is right		
		if( same_fields && is_address_increment_right(m_request_buffer[current_index], m_request_buffer[beginning_burst_index]) )  {
			new_chunk_transaction=true;
			
			if(max_chunk_length==1)
				single=true;
			else single=false;

			//check if the burst is finished
			if(is_last_of_burst(m_request_buffer[current_index], single))   {
				new_burst_transaction=true;
				
			}
			else new_burst_transaction=false;

			return SEND_CHUNK_WITH_REQUEST;
		}
		else {
			//the fields have changed, or the address increment is wrong
			//check if there is a SINGLE request
			if(is_last_of_burst(m_request_buffer[current_index], false))  {
				new_burst_transaction=new_chunk_transaction=true;
				return SEND_CHUNK_AND_SINGLE_REQUEST;
			}
			//or not: send current burst, store new request
			else {
				new_burst_transaction=new_chunk_transaction=false;
				return SEND_CHUNK_WITHOUT_REQUEST;
			}
		}		
	}
	else  {
		//this is the beginning of a new transaction
		if(new_burst_transaction)  {
			//for precise burst, set the remaining number of requests in the burst
			if(burst_precise)
                precise_burst_length[current_index]=m_request_buffer[current_index].MBurstLength-1;
		
			//check if there is a SINGLE request
			if(is_last_of_burst(m_request_buffer[current_index], true))   {
				new_chunk_transaction=new_burst_transaction=true;
				return SEND_CHUNK_WITH_REQUEST;
			}
			//or not: store new request along with current burst
			else {
				new_chunk_transaction=new_burst_transaction=false;
				return STORE_NEW_REQUEST;
			}
		}
		else {
			//the request fields are the same, and address increment is right		
			if( same_fields && is_address_increment_right(m_request_buffer[current_index], m_request_buffer[beginning_burst_index]) )  {
				//the fields are the same:
				if(same_other_fields)  {
					//check if this is the last of the burst
					if(is_last_of_burst(m_request_buffer[current_index], false))   {
						new_burst_transaction=new_chunk_transaction=true;
						return SEND_CHUNK_WITH_REQUEST;
					}
					//or not: store new request along with current burst
					else {
						new_burst_transaction=new_chunk_transaction=false;
						return STORE_NEW_REQUEST;	
					}
				}	
				//the fields have changed, new chunk:
				else   {
					new_chunk_transaction=true;
					//check if this is the last of the burst: sends the previous chunk with the chunk composed of one request 
					if(is_last_of_burst(m_request_buffer[current_index], false))   {
						new_burst_transaction=true;
						return SEND_CHUNK_AND_SINGLE_REQUEST;
					}
					//or not: store new request along with current burst
					else {
						new_burst_transaction=false;
						return SEND_CHUNK_WITHOUT_REQUEST;	
					}
				}
			}
			else {
				//the fields have changed, or the address increment is wrong : current request is the beginning of a new burst
				//this is the beginning of a new burst: for precise burst, get the new length of the burst
				if(burst_precise)
                    precise_burst_length[current_index]=precise_burst_length[current_index-1]-1;

				//check if there is a SINGLE request
				if(is_last_of_burst(m_request_buffer[current_index], false))  	{
					new_burst_transaction=new_chunk_transaction=true;
					return SEND_CHUNK_AND_SINGLE_REQUEST;
				}
				//or not: send current burst, store new request
				else {
					new_burst_transaction=false;
					new_chunk_transaction=true;
					return SEND_CHUNK_WITHOUT_REQUEST;	
				}
			}
		}
	}
}


//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter::same_request_fields
//
//---------------------------------------------------------------------------
template<class TdataCl_tl1>
bool OCP_TL1_TL2_Master_Adapter<TdataCl_tl1>::same_request_fields(OCPRequestGrp<Td, Ta>& new_req_param, OCPRequestGrp<Td, Ta>& current_req_param )  {

	//check if the fields have changed 
	//some fields can change during the burst: MAddr, MByteEn, MDataInfo, MReqLast
	//Multiple threads are not supported (hence MThreadID variation ends simulation)
	
	if(new_req_param.MThreadID != current_req_param.MThreadID) {
		cerr<<"\n**************************** TL1-TL2 Master Adapter \""<<name()<< "\" :";
		cerr<<" Error\n";
		cerr<<"Multiple threads are not supported\n****************************"<<endl;
		sc_stop();
	}
	else if(new_req_param.MCmd != current_req_param.MCmd) 
		return false;
	else if(new_req_param.MAddrSpace != current_req_param.MAddrSpace)
		return false;
	else if(new_req_param.MConnID != current_req_param.MConnID)
		return false;
	else if(new_req_param.MReqInfo != current_req_param.MReqInfo)
		return false;
	else if(new_req_param.MAtomicLength != current_req_param.MAtomicLength)
		return false;
	//tests the other fields relative to burst support, if necessary 
	else if(burst_support) {
		if(new_req_param.MBurstPrecise != current_req_param.MBurstPrecise)
			return false;
		else if(new_req_param.MBurstSeq != current_req_param.MBurstSeq)
			return false;
		else if(burst_precise)  {
			if(new_req_param.MBurstLength != current_req_param.MBurstLength)
				return false;
		}
	}
		
	//fields have not changed:
	return true;
}

//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter::same_other_request_fields
//fields MByteEn or MDataInfo have changed: a new chunk begins
//---------------------------------------------------------------------------
template<class TdataCl_tl1>
bool OCP_TL1_TL2_Master_Adapter<TdataCl_tl1>::same_other_request_fields(OCPRequestGrp<Td, Ta>& new_req_param, OCPRequestGrp<Td, Ta>& current_req_param )  {
	if(new_req_param.MByteEn != current_req_param.MByteEn) 
		return false;
	else if(new_req_param.MDataInfo != current_req_param.MDataInfo)
		return false;

	//fields have not changed:
	return true;
}


//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter::is_address_increment_right
//
//---------------------------------------------------------------------------
template<class TdataCl_tl1>
bool OCP_TL1_TL2_Master_Adapter<TdataCl_tl1>::is_address_increment_right(OCPRequestGrp<Td, Ta>& new_req_param, OCPRequestGrp<Td, Ta>& current_req_param )  {

	if(burst_support)  {
		//if burst is a streaming burst, verify if the address has not changed
		if(new_req_param.MBurstSeq==OCP_MBURSTSEQ_STRM)  {	
			if( (current_req_param.MBurstSeq==OCP_MBURSTSEQ_STRM) && (new_req_param.MAddr ==current_req_param.MAddr))
				return true;
			else return false;
		}
		//if burst is a incremental burst, verify if the address increment has changed
		else if(new_req_param.MBurstSeq==OCP_MBURSTSEQ_INCR) {
			if(current_index<beginning_burst_index) {
				if(new_req_param.MAddr==current_req_param.MAddr+(current_index+buffer_size-beginning_burst_index)*m_NumBytesPerWord)	
					return true;
				else return false;
			}
			else {
				if(new_req_param.MAddr==current_req_param.MAddr+(current_index-beginning_burst_index)*m_NumBytesPerWord)
					return true;
				else return false;
			}					
		}
	}
	else  {
		//no burst support: verify if the address increment is right
		if(current_index<beginning_burst_index) {
			if(new_req_param.MAddr==current_req_param.MAddr+(current_index+buffer_size-beginning_burst_index)*m_NumBytesPerWord)	
				return true;
			else return false;
			}
		else {
			if(new_req_param.MAddr==current_req_param.MAddr+(current_index-beginning_burst_index)*m_NumBytesPerWord)
				return true;
			else return false;
		}
	}


	//in the other cases (custom bursts), send as single TL2 requests 
	return false;
}


//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter::reset_request_parameters
//reset request parameters
//---------------------------------------------------------------------------
template<class TdataCl_tl1>
void OCP_TL1_TL2_Master_Adapter<TdataCl_tl1>::reset_request_parameters(int i)   {
	//calls the reset function of OCPRequestGrp class
	m_request_buffer[i].reset(false);
}


//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter::is_last_of_burst
//indicates if it is the last TL1 request of the burst
//---------------------------------------------------------------------------
template<class TdataCl_tl1>
bool OCP_TL1_TL2_Master_Adapter<TdataCl_tl1>::is_last_of_burst(OCPRequestGrp<Td, Ta>& new_req_param, bool new_burst)   {
	//no burst support: test if MReqLast is true 
	if(!burst_support)   {
		if(new_req_param.MReqLast==true)
			return true;
		else return false;
	}
	else {
		//imprecise burst
		if(!burst_precise)   {
			//check if MBurstLength=1, which indicates the end of the burst
			if(new_req_param.MBurstLength==1)  
				return true;
			else return false;
		}
		//precise burst: check if burst is finished 
		else {  
			if(!new_burst)  {
				if(current_index>0)
					precise_burst_length[current_index]=precise_burst_length[current_index-1]-1;
				else precise_burst_length[0]=precise_burst_length[buffer_size-1]-1;	
				
				if(precise_burst_length[current_index]==0)  
					return true;		
				else return false;
			}
			else {
				if(m_request_buffer[current_index].MBurstLength==1)
					return true;
				else return false;
			}
        }
	}	
}


//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter::timeout_activated 
//no request or response has arrived during the m_timeout time
//---------------------------------------------------------------------------
template<class TdataCl_tl1>
void OCP_TL1_TL2_Master_Adapter<TdataCl_tl1>::timeout_activated()   {
	cout<<"\n\n****************************\n TL1-TL2 Master Adapter \""<<name()<< "\" :";
	cout<<" Warning (t="<<sc_time_stamp()<<") \n";
	cout<<"Timeout for TL1-TL2 Master adapter \""<<name()<<"\"\n***************************\n"<<endl;
}


// ----------------------------------------------------------------------------
//  Method : OCP_TL1_TL2_Master_Adapter::end_of_elaboration()
//
//  This method is activated at the end of the elaboration phase
//  (when all binding operations are finished).
// ----------------------------------------------------------------------------

template<class TdataCl_tl1>
void OCP_TL1_TL2_Master_Adapter<TdataCl_tl1>::end_of_elaboration()
{

  sc_module::end_of_elaboration();

  // Get data structure
  m_DataCl_tl1 = SlaveP->GetDataCl();

  // Get system parameter structure
  m_ParamCl = SlaveP->GetParamCl();

  //multiple threads are not supported
  if(m_ParamCl->threads>1)  {
	  cerr<<"\n****************************\nTL1-TL2 Master Adapter \""<<name()<< "\" :";
	  cerr<<"\nError: ";
	  cerr<<"Multiple threads are not supported\n****************************"<<endl;
	  sc_stop();
  }

  //check for burst support
  if((m_ParamCl->burstseq)&&(m_ParamCl->burstlength))   {
	  burst_support=true;
	  //check if bursts are precise
	  if(m_ParamCl->burstprecise)
		  burst_precise=true;
	  else burst_precise=false;
  }
  else burst_support=false;

  if(burst_precise)  {
	precise_burst_length=new unsigned int[buffer_size];
	for(int i=0; i<buffer_size; i++)   
		precise_burst_length[i]=0;

  }

  //check for threadbusy compliances
  if((m_ParamCl->mthreadbusy_exact) || (m_ParamCl->sthreadbusy_exact))  {
	  cerr<<"\n****************************\nTL1-TL2 Master Adapter \""<<name()<< "\" :";
	  cerr<<"\nError: ";
	  cerr<<"Threadbusy compliance is not supported\n****************************"<<endl;
	  sc_stop();
  }

  //check for single request/multiple data
  if((m_ParamCl->burstsinglereq))  {
	  cerr<<"\n****************************\nTL1-TL2 Master Adapter \""<<name()<< "\" :";
	  cerr<<"\nError: ";
	  cerr<<"Single Request/Multiple Data compliance is not supported\n****************************"<<endl;
	  sc_stop();
  }

  //check for datahandshake 
  if((m_ParamCl->datahandshake))  {
	  cerr<<"\n****************************\nTL1-TL2 Master Adapter \""<<name()<< "\" :";
	  cerr<<"\nError: ";
	  cerr<<"Single Request/Multiple Data compliance is not supported\n****************************"<<endl;
	  sc_stop();
  }

  //check for cmdaccept 
  if(!m_ParamCl->cmdaccept)  {
	  cerr<<"\n****************************\nTL1-TL2 Master Adapter \""<<name()<< "\" :";
	  cerr<<"\nError: ";
	  cerr<<"cmdaccept parameters must be set\n****************************"<<endl;
	  sc_stop();
  }

  //check for srespaccept 
  if(!m_ParamCl->respaccept)  {
	  cerr<<"\n****************************\nTL1-TL2 Master Adapter \""<<name()<< "\" :";
	  cerr<<"\nError: ";
	  cerr<<"respaccept parameters must be set\n****************************"<<endl;
	  sc_stop();
  }

  // Put parameter into Channel
  // No parameters needed

}


// ----------------------------------------------------------------------------
//
//  Instantiation of the adapter
//
// ----------------------------------------------------------------------------

template class OCP_TL1_TL2_Master_Adapter<OCP_TL1_DataCl<int , int> >; 

