// 
//  (c) Copyright OCP-IP 2003, 2004
//  OCP-IP Confidential and Proprietary
//
// ============================================================================
//      Project : OCP SLD WG, Layer adapter examples 
//      Authors : Yann Bajot, PROSILOG, bajot@prosilog.com 
//                Stephane Guntz, PROSILOG, guntz@prosilog.com
//
//  Description : Transaction Level - Layer-1 to Layer-2 Master Adapter (OCP 2.0 supported)
//  $Id: ocp_tl1_tl2_master_adapter.h,v 1.1.1.1 2004/09/24 09:26:21 sguntz Exp $
//
//  Modifications (09/01/2004): 
//		- correction of array indexing,
//		- correct bug for computing of address_increment and making relevant TL2 request chunks (occured when master adapter receives more than adapter_depth TL2 chunks)
//		- correct bug when a response chunk is being processed, the buffer is full and a new one is received,
//		- add master adapter name in debug printing messages
//		- change some debug messages
//
//	Modifications (09/22/2004): new version which is compatible with both new TL2 performance channel and old layered TL2 channel
//	    - remove TL2 dataclass as a template (only one template now: the TL1 dataclass)
//	    - remove references to direct interfaces 
//    
//  Parameters    :
//    Template arguments.
//     - TdataCl_tl1: TL1 Data class containing data members and
//                access methods for the data exchanged between
//                Masters and Slaves
//     - Td: Data type
//     - Ta: Address type
//                
//   Constructor arguments:
//      - sc_module_name: Name of the module instance
//	    - max_chunk_length: maximum size for a chunk of a burst
//		- adapter_depth: Number of missed events and responses to be cached.
//						 The adapter stores new request/response infos which have arrived during the processing of the current request/response
//						 and sends them when the current request/response is finished
//		- timeout: a warning is sent if during a timeout period, no TL1 request or
//		TL2 response has been received by the master adapter
//	 
//	 
//  The TL1 master adapter has a TL2 master interface on one side, and a TL1 slave interface on the other side.
//  It must be connected to:
//      - a TL1 master through a TL1 channel
//      - a TL2 slave through a TL2 channel
//
//	The adapter retrieves TL1 requests and stores them in its internal buffer. When a new request arrived, the adapter
//	compares it with former requests to decide whether it is part of the current burst or the beginning of a new burst
//  (it can be a single request too).
//	When a burst is finished, a TL2 request is sent to the slave.
//  
//  On the response side, the TL1 master adapter retrieves the TL2 response from the slave, and sends corresponding
//  TL1 responses to the master. 
//
//	The following is unsupported:
//	- Single request/multiple data
//	- Data handshake
//	- Multiple threads
//	- SThreadBusy/MThreadBusy compliances (m_threadbusy_exact=0, s_threadbusy_exact=0)
//  
// ============================================================================

#ifndef _OCP_TL1_TL2_MASTER_ADAPTER_H
#define _OCP_TL1_TL2_MASTER_ADAPTER_H

#include "systemc.h"
#include "ocp_globals.h"
#include "ocp_tl1_data_cl.h"
#include "ocp_tl1_slave_port.h"
#include "ocp_tl2_master_port.h"
#include "ocp_tl_param_cl.h"


//for debugging purposes, print messages on screen
//comment for no messages
#define DEBUG_MASTER_ADAPTER_TL1


#ifndef MASTER_ADAPTER_TASK_ENUM
#define MASTER_ADAPTER_TASK_ENUM

enum MasterAdapterTask   {
	SEND_CHUNK_WITH_REQUEST = 0,
	SEND_CHUNK_WITHOUT_REQUEST,
	STORE_NEW_REQUEST,
	SEND_CHUNK_AND_SINGLE_REQUEST
};

#endif


template <class TdataCl_tl1> class OCP_TL1_TL2_Master_Adapter
  : public sc_module  
{
public:  
  //TL1 data type and address type
  typedef typename TdataCl_tl1::DataType Td;
  typedef typename TdataCl_tl1::AddrType Ta;
  
  // TL1 Slave Port and TL2 Master port
  OCP_TL1_SlavePort<TdataCl_tl1 > SlaveP;
  OCP_TL2_MasterPort<Td, Ta > MasterP;

  //clock port
  sc_in_clk clk;

  SC_HAS_PROCESS(OCP_TL1_TL2_Master_Adapter);

  // constructor
  OCP_TL1_TL2_Master_Adapter(sc_module_name name, int max_chunk_length, int adapter_depth, sc_time timeout);

  // destructor
  ~OCP_TL1_TL2_Master_Adapter();

  // SystemC processes
  void MasterRequest();
  void SlaveRequest();
  void MasterResponse();
  void SlaveResponse();
  void timeout_activated();
  
  
  //adapter methods
  void reset_request_parameters(int);
  bool same_request_fields(OCPRequestGrp<Td, Ta>&, OCPRequestGrp<Td, Ta>&);
  bool same_other_request_fields(OCPRequestGrp<Td, Ta>&, OCPRequestGrp<Td, Ta>&);
  bool is_address_increment_right(OCPRequestGrp<Td, Ta>&, OCPRequestGrp<Td, Ta>&);
  bool is_last_of_burst(OCPRequestGrp<Td, Ta>&, bool);
  MasterAdapterTask define_adapter_task(void);

 private :
  void end_of_elaboration();

  // pointer to the TL1 data structure of the channel
  TdataCl_tl1          *m_DataCl_tl1;
  
  //pointer to param class
  ParamCl<TdataCl_tl1> *m_ParamCl; 
       
  // parameters given as constructor parameters
  int max_chunk_length;        //maximum length for a chunk of a burst transaction
  int buffer_size;             //size of the internal buffer for data and request fields (MCmd, MAddr, MAddrSpace...)
  int adapter_depth;		  //depth of requests and response storage 

  //internal parameters
  bool new_burst_transaction;  //indicates if the next transaction is the beginning of a new burst
  bool new_chunk_transaction;  //indicates if the next transaction is the beginning of a new chunk
  bool last_of_a_burst;		   // last datum of the burst is sent with the burst
  int m_NumBytesPerWord;	   // size of the data
  bool burst_support;			//does the channel has burst support or not
  bool burst_precise;			//bursts are precise or not
  sc_time m_timeout;			//time after which the channel is released when it is blocked for expecting a new request
  
  //indicates a later release of the request or response thread, when the internal buffer for storing request/response is full
  bool release_request_thread; 
  bool release_response_thread;
  bool request_not_finished;

  //index of the currently processed request/response
  int current_processed_request;
  int current_processed_response;
  
  int beginning_burst_index, current_index;   //index for internal buffers
  
  //buffers for storing beginning index and length of request/response
  int* burst_request_index;		//beginning index of the TL2 request
  int* burst_request_length;	//length of the TL2 request
  int* response_index;	//beginning index of the TL1 response
  unsigned int* response_length;	//number of TL1 responses to send
  unsigned int* precise_burst_length; //remaining number of requests, for precise bursts

  int m_pending_request;   //indicates if there is a pending request which has arrived, during the processing of the current request
  int m_request_number;	   //ID of the request currently processed
  int m_pending_response;  //indicates if there is a pending response which has arrived, during the processing of the current request
  int m_response_number;   //ID of the response currently processed

  //internal buffers
  OCPRequestGrp<Td, Ta>* m_request_buffer; //for storing TL1 request fields
  OCPResponseGrp<Td>* m_response_buffer;   //for storing TL2 response fields
  Td* m_data;							//for storing data
  Td* m_response_data;					//for storing response
  
  //events
  sc_event send_TL2_request_event;	//event for sending TL2 request, made of TL1 atomic requests
  sc_event send_TL1_response_event; //event for sending TL1 responses, made of TL2 response
  sc_event thread_released_event;   //event indicating the release of the request thread
  sc_event timeout_event;
};


#endif // _OCP_TL1_TL2_MASTER_ADAPTER_H
